/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.gui.menu;

import java.awt.event.KeyEvent;

import java.beans.PropertyChangeEvent;

import javax.swing.JMenuItem;

import net.sf.jaxodraw.util.JaxoConstants;

/** Contains the entries of the edit menu.
 * @since 2.0
 */
public class JaxoEditMenu extends JaxoMenu {
    private static final long serialVersionUID = 7526471155622776147L;

    private final JMenuItem undoMI = new JMenuItem();
    private final JMenuItem redoMI = new JMenuItem();

    /** Constructs a new edit menu from the specified panel. */
    public JaxoEditMenu() {
        super();

        this.add(undoMenuItem());
        this.add(redoMenuItem());
        this.add(clearMenuItem());
        this.add(pasteMenuItem());

        addSeparator();

        this.add(refreshMenuItem());

        addSeparator();

        this.add(moveMenuItem());
        this.add(resizeMenuItem());
        this.add(duplicateMenuItem());
        this.add(colorMenuItem());
        this.add(editMenuItem());
        this.add(deleteMenuItem());

        addSeparator();

        this.add(backgroundMenuItem());
        this.add(foregroundMenuItem());

        addSeparator();

        this.add(selectMenuItem());
        this.add(ungroupMenuItem());

        addSeparator();

        this.add(zoomMenuItem());

        JaxoMenuUtils.setLanguageKey(this, "Edit");
        updateLanguage();
    }

    /**
     * Applies a property change event.
     *
     * @param e the change event.
     */
    public void propertyChange(final PropertyChangeEvent e) {
        final String name = e.getPropertyName();

        if ("Jaxo.canUndo".equals(name)) {
            undoMI.setEnabled(Boolean.TRUE.equals(e.getNewValue()));
        } else if ("Jaxo.canRedo".equals(name)) {
            redoMI.setEnabled(Boolean.TRUE.equals(e.getNewValue()));
        }
    }

      //
     // private methods
    //

    private JMenuItem zoomMenuItem() {
        final JMenuItem zoomMI = new JMenuItem();
        JaxoMenuUtils.setAccelerator(zoomMI, KeyEvent.VK_Z, ALT_MASK);
        JaxoMenuUtils.setActionCommand(zoomMI, JaxoConstants.ZOOM);
        JaxoMenuUtils.setLanguageKey(zoomMI, "Zoom");
        return zoomMI;
    }

    private JMenuItem ungroupMenuItem() {
        final JMenuItem ungroupMI = new JMenuItem();
        JaxoMenuUtils.setAccelerator(ungroupMI, KeyEvent.VK_N, ALT_MASK);
        JaxoMenuUtils.setActionCommand(ungroupMI, JaxoConstants.UNGROUP);
        JaxoMenuUtils.setLanguageKey(ungroupMI, "Ungroup");
        return ungroupMI;
    }

    private JMenuItem selectMenuItem() {
        final JMenuItem selectMI = new JMenuItem();
        JaxoMenuUtils.setAccelerator(selectMI, KeyEvent.VK_G, ALT_MASK);
        JaxoMenuUtils.setActionCommand(selectMI, JaxoConstants.SELECT);
        JaxoMenuUtils.setLanguageKey(selectMI, "Select");
        return selectMI;
    }

    private JMenuItem foregroundMenuItem() {
        final JMenuItem foregroundMI = new JMenuItem();
        JaxoMenuUtils.setAccelerator(foregroundMI, KeyEvent.VK_F, ALT_MASK);
        JaxoMenuUtils.setActionCommand(foregroundMI, JaxoConstants.FOREGROUND);
        JaxoMenuUtils.setLanguageKey(foregroundMI, "Foreground");
        return foregroundMI;
    }

    private JMenuItem backgroundMenuItem() {
        final JMenuItem backgroundMI = new JMenuItem();
        JaxoMenuUtils.setAccelerator(backgroundMI, KeyEvent.VK_B, ALT_MASK);
        JaxoMenuUtils.setActionCommand(backgroundMI, JaxoConstants.BACKGROUND);
        JaxoMenuUtils.setLanguageKey(backgroundMI, "Background");
        return backgroundMI;
    }

    private JMenuItem deleteMenuItem() {
        final JMenuItem deleteMI = new JMenuItem();
        JaxoMenuUtils.setAccelerator(deleteMI, KeyEvent.VK_D, ALT_MASK);
        JaxoMenuUtils.setActionCommand(deleteMI, JaxoConstants.DELETE);
        JaxoMenuUtils.setLanguageKey(deleteMI, "Delete");
        return deleteMI;
    }

    private JMenuItem editMenuItem() {
        final JMenuItem editMI = new JMenuItem();
        JaxoMenuUtils.setAccelerator(editMI, KeyEvent.VK_E, ALT_MASK);
        JaxoMenuUtils.setActionCommand(editMI, JaxoConstants.EDIT);
        JaxoMenuUtils.setLanguageKey(editMI, "Edit");
        return editMI;
    }

    private JMenuItem colorMenuItem() {
        final JMenuItem colorMI = new JMenuItem();
        JaxoMenuUtils.setAccelerator(colorMI, KeyEvent.VK_L, ALT_MASK);
        JaxoMenuUtils.setActionCommand(colorMI, JaxoConstants.COLOR);
        JaxoMenuUtils.setLanguageKey(colorMI, "Color");
        return colorMI;
    }

    private JMenuItem duplicateMenuItem() {
        final JMenuItem duplicateMI = new JMenuItem();
        JaxoMenuUtils.setAccelerator(duplicateMI, KeyEvent.VK_C, ALT_MASK);
        JaxoMenuUtils.setActionCommand(duplicateMI, JaxoConstants.DUPLICATE);
        JaxoMenuUtils.setLanguageKey(duplicateMI, "Duplicate");
        return duplicateMI;
    }

    private JMenuItem resizeMenuItem() {
        final JMenuItem resizeMI = new JMenuItem();
        JaxoMenuUtils.setAccelerator(resizeMI, KeyEvent.VK_R, ALT_MASK);
        JaxoMenuUtils.setActionCommand(resizeMI, JaxoConstants.RESIZE);
        JaxoMenuUtils.setLanguageKey(resizeMI, "Resize");
        return resizeMI;
    }

    private JMenuItem moveMenuItem() {
        final JMenuItem moveMI = new JMenuItem();
        JaxoMenuUtils.setAccelerator(moveMI, KeyEvent.VK_M, ALT_MASK);
        JaxoMenuUtils.setActionCommand(moveMI, JaxoConstants.MOVE);
        JaxoMenuUtils.setLanguageKey(moveMI, "Move");
        return moveMI;
    }

    private JMenuItem refreshMenuItem() {
        final JMenuItem refreshMI = new JMenuItem();
        JaxoMenuUtils.setAccelerator(refreshMI, KeyEvent.VK_S, ALT_MASK);
        JaxoMenuUtils.setActionCommand(refreshMI, JaxoConstants.REFRESH);
        JaxoMenuUtils.setLanguageKey(refreshMI, "Refresh");
        return refreshMI;
    }

    private JMenuItem pasteMenuItem() {
        final JMenuItem pasteMI = new JMenuItem();
        JaxoMenuUtils.setAccelerator(pasteMI, KeyEvent.VK_V, CTRL_MASK);
        JaxoMenuUtils.setActionCommand(pasteMI, JaxoConstants.PASTE);
        JaxoMenuUtils.setLanguageKey(pasteMI, "Paste");
        return pasteMI;
    }

    private JMenuItem clearMenuItem() {
        final JMenuItem clearMI = new JMenuItem();
        JaxoMenuUtils.setAccelerator(clearMI, KeyEvent.VK_X, ALT_MASK);
        JaxoMenuUtils.setActionCommand(clearMI, JaxoConstants.CLEAR);
        JaxoMenuUtils.setLanguageKey(clearMI, "Clear");
        return clearMI;
    }

    private JMenuItem redoMenuItem() {
        JaxoMenuUtils.setAccelerator(redoMI, KeyEvent.VK_Y, CTRL_MASK);
        JaxoMenuUtils.setActionCommand(redoMI, JaxoConstants.REDO);
        JaxoMenuUtils.setLanguageKey(redoMI, "Redo");
        redoMI.setEnabled(false);
        return redoMI;
    }

    private JMenuItem undoMenuItem() {
        JaxoMenuUtils.setAccelerator(undoMI, KeyEvent.VK_Z, CTRL_MASK);
        JaxoMenuUtils.setActionCommand(undoMI, JaxoConstants.UNDO);
        JaxoMenuUtils.setLanguageKey(undoMI, "Undo");
        undoMI.setEnabled(false);
        return undoMI;
    }
}
