/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.gui.menu;

import javax.swing.ImageIcon;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JMenuItem;
import javax.swing.KeyStroke;

import net.sf.jaxodraw.util.JaxoConstants;
import net.sf.jaxodraw.util.JaxoLanguage;
import net.sf.jaxodraw.util.JaxoUtils;

/**
 * Menu-related utility methods.
 */
public final class JaxoMenuUtils {

    // for now, a compile-time option, could be added to the preferences
    private static final boolean USE_CUSTOM_CHECKMARK = true;
    private static final ImageIcon CHECKMARK_ICON;
    private static final String I18N_KEY = "Jaxo.i18NKey";

    static {
        CHECKMARK_ICON = JaxoUtils.newImageIcon("checkmark.png");
    }

    private JaxoMenuUtils() {
        // empty on purpose
    }

    /**
     * =newCheckmarMenuItem(null).
     * @return A new JMenuItem.
     */
    public static JMenuItem newCheckmarkMenuItem() {
        return USE_CUSTOM_CHECKMARK ? new JMenuItem()
                                    : new JCheckBoxMenuItem();
    }

    /**
     * Depending on configuration, a JCheckBoxMenuItem or a JMenuItem with custom Icon.
     *        Use {#isChecked}, {#setChecked} to control checking instead of is/setSelected.
     * @param text The text of the JMenuItem.
     * @return A new JMenuItem.
     */
    public static JMenuItem newCheckmarkMenuItem(final String text) {
        return USE_CUSTOM_CHECKMARK ? new JMenuItem(text)
                                    : new JCheckBoxMenuItem(text);
    }

    /**
     * Checks whether the given JMenuItem is selected.
     * @param m The JMenuItem to check.
     * @return True if the JMenuItem is selected.
     * @see #newCheckmarkMenuItem
     */
    public static boolean isChecked(final JMenuItem m) {
        if (USE_CUSTOM_CHECKMARK) {
            return m.getIcon() != null;
        } else {
            return m.isSelected();
        }
    }

    /**
     * Sets the given JMenuItem as selected.
     * @param m The JMenuItem to check.
     * @param value True to set the JMenuItem as selected.
     * @see #newCheckmarkMenuItem
     */
    public static void setChecked(final JMenuItem m, final boolean value) {
        if (USE_CUSTOM_CHECKMARK) {
            m.setIcon(value ? CHECKMARK_ICON : null);
        } else {
            m.setSelected(value);
        }
    }

    /**
     * Update the text of a JMenuItem.
     *
     * @param m The JMenuItem.
     */
    public static void updateLanguage(final JMenuItem m) {
        if (m.getClientProperty(I18N_KEY) != null) {
            m.setText(JaxoLanguage.translate((String) m.getClientProperty(I18N_KEY)));
        }
    }

    /**
     * Set the i18n key of a JMenuItem.
     *
     * @param m The JMenuItem.
     * @param key the language key.
     */
    public static void setLanguageKey(final JMenuItem m, final String key) {
        m.putClientProperty(I18N_KEY, key);
    }

    /**
     * Sets the Keyboard Accelerator of a JMenuItem to the specified character.
     *
     * @param m The JMenuItem.
     * @param keyChar the character value for a keyboard key
     * @see javax.swing.KeyStroke#getKeyStroke(char)
     */
    public static void setAccelerator(final JMenuItem m, final char keyChar) {
        m.setAccelerator(KeyStroke.getKeyStroke(keyChar));
    }

    /**
     * Sets the Keyboard Accelerator of a JMenuItem to a KeyStroke
     * given by a numeric key code and a set of modifiers.
     *
     * @param m The JMenuItem.
     * @param keyCode an int specifying the numeric code for a keyboard key.
     * @param modifiers a bitwise-ored combination of any modifiers.
     * @see javax.swing.KeyStroke#getKeyStroke(int,int)
     */
    public static void setAccelerator(final JMenuItem m, final int keyCode, final int modifiers) {
        m.setAccelerator(KeyStroke.getKeyStroke(keyCode, modifiers));
    }

    /**
     * Sets the ActionCommand of a JMenuItem to the given mode.
     *
     * @param m The JMenuItem.
     * @param mode A mode as defined in JaxoConstants.
     * @see net.sf.jaxodraw.util.JaxoConstants#getModeAsString(int)
     */
    public static void setActionCommand(final JMenuItem m, final int mode) {
        m.setActionCommand(JaxoConstants.getModeAsString(mode));
    }

}
