/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.gui.menu;

import java.awt.event.KeyEvent;

import java.beans.PropertyChangeEvent;
import javax.swing.JMenuItem;

import net.sf.jaxodraw.util.JaxoConstants;
import net.sf.jaxodraw.util.JaxoPrefs;

/** Contains the entries of the options menu.
 * @since 2.0
 */
public class JaxoOptionsMenu extends JaxoMenu {
    private static final long serialVersionUID = 7526471155622776147L;

    private final JMenuItem showtoolMI
            = JaxoMenuUtils.newCheckmarkMenuItem();
    private final JMenuItem showStatusMI
            = JaxoMenuUtils.newCheckmarkMenuItem();
    private final JMenuItem showgridMI
            = JaxoMenuUtils.newCheckmarkMenuItem();
    private final JMenuItem antialiasOnMI
            = JaxoMenuUtils.newCheckmarkMenuItem();
    private final JMenuItem gridOnMI
            = JaxoMenuUtils.newCheckmarkMenuItem();
    private final JMenuItem snapOnMI
            = JaxoMenuUtils.newCheckmarkMenuItem();
    private final JMenuItem arrowMI
            = JaxoMenuUtils.newCheckmarkMenuItem();

    /** Constructs a new options menu from the specified panel. */
    public JaxoOptionsMenu() {
        super();

        this.add(describeMenuItem());
        this.add(packageMenuItem());
        this.add(moveGraphItem());

        addSeparator();

        this.add(lookAndFeelMenuItem());
        this.add(languageMenu());
        this.add(defaultMenu());

        addSeparator();

        this.add(vertexMenu());

        addSeparator();

        this.add(showtoolMenuItem());
        this.add(showstatusMenuItem());
        this.add(showgridMenuItem());
        this.add(antialiasOnMenuItem());
        this.add(arrowMenuItem());
        this.add(gridOnMenuItem());
        this.add(snapOnMenuItem());

        addSeparator();

        this.add(pluginManagerMenuItem());

        addSeparator();

        this.add(preferencesMenuItem());

        JaxoMenuUtils.setLanguageKey(this, "Options");
        updateLanguage();
    }

    /**
     * Applies a property change event.
     *
     * @param e the change event.
     */
    public void propertyChange(final PropertyChangeEvent e) {
        final String name = e.getPropertyName();

        if ("Jaxo.toolBarVisible".equals(name)) {
            JaxoMenuUtils.setChecked(showtoolMI,
                Boolean.TRUE.equals(e.getNewValue()));
        } else if ("Jaxo.statusBarVisible".equals(name)) {
            JaxoMenuUtils.setChecked(showStatusMI,
                Boolean.TRUE.equals(e.getNewValue()));
        } else if ("Jaxo.gridBarVisible".equals(name)) {
            JaxoMenuUtils.setChecked(showgridMI,
                Boolean.TRUE.equals(e.getNewValue()));
        } else if ("Jaxo.gridOn".equals(name)) {
            JaxoMenuUtils.setChecked(gridOnMI, Boolean.TRUE.equals(e.getNewValue()));
        } else if ("Jaxo.snap".equals(name)) {
            JaxoMenuUtils.setChecked(snapOnMI, Boolean.TRUE.equals(e.getNewValue()));
        } else if ("Jaxo.antialiasEnabled".equals(name)) {
            JaxoMenuUtils.setChecked(antialiasOnMI,
                Boolean.TRUE.equals(e.getNewValue()));
        } else if ("Jaxo.withArrow".equals(name)) {
            JaxoMenuUtils.setChecked(arrowMI, Boolean.TRUE.equals(e.getNewValue()));
        } else {
            // dispatch to sub-menus
            firePropertyChange(name, e.getOldValue(), e.getNewValue());
        }
    }

      //
     // private methods
    //

    private JMenuItem preferencesMenuItem() {
        final JMenuItem preferencesMI = new JMenuItem();
        JaxoMenuUtils.setAccelerator(preferencesMI, KeyEvent.VK_U, DEFAULT_MASK);
        JaxoMenuUtils.setActionCommand(preferencesMI, JaxoConstants.PREFERENCES);
        JaxoMenuUtils.setLanguageKey(preferencesMI, "Preferences");
        return preferencesMI;
    }

    private JMenuItem pluginManagerMenuItem() {
        final JMenuItem pluginsMI = new JMenuItem();
        //JaxoMenuUtils.setAccelerator(pluginsMI, KeyEvent.VK_U, DEFAULT_MASK);
        JaxoMenuUtils.setActionCommand(pluginsMI, JaxoConstants.PLUGINS);
        JaxoMenuUtils.setLanguageKey(pluginsMI, "PluginManager");
        return pluginsMI;
    }

    private JMenuItem snapOnMenuItem() {
        JaxoMenuUtils.setChecked(snapOnMI,
            JaxoPrefs.getBooleanPref(JaxoPrefs.PREF_SNAPONOFF));
        JaxoMenuUtils.setActionCommand(snapOnMI, JaxoConstants.SNAP_ONOFF);
        JaxoMenuUtils.setLanguageKey(snapOnMI, "Snap_on");
        return snapOnMI;
    }

    private JMenuItem gridOnMenuItem() {
        JaxoMenuUtils.setChecked(gridOnMI,
            JaxoPrefs.getBooleanPref(JaxoPrefs.PREF_GRIDONOFF));
        JaxoMenuUtils.setActionCommand(gridOnMI, JaxoConstants.GRID_ONOFF);
        JaxoMenuUtils.setLanguageKey(gridOnMI, "Grid_on");
        return gridOnMI;
    }

    private JMenuItem arrowMenuItem() {
        JaxoMenuUtils.setChecked(arrowMI,
            JaxoPrefs.getBooleanPref(JaxoPrefs.PREF_ARROW));
        JaxoMenuUtils.setActionCommand(arrowMI, JaxoConstants.ARROW);
        JaxoMenuUtils.setLanguageKey(arrowMI, "Arrow");
        return arrowMI;
    }

    private JMenuItem antialiasOnMenuItem() {
        JaxoMenuUtils.setChecked(antialiasOnMI,
            JaxoPrefs.getBooleanPref(JaxoPrefs.PREF_ANTIALIAS));
        JaxoMenuUtils.setActionCommand(antialiasOnMI, JaxoConstants.ANTIALIAS);
        JaxoMenuUtils.setLanguageKey(antialiasOnMI, "Antialias_on");
        return antialiasOnMI;
    }

    private JMenuItem showgridMenuItem() {
        JaxoMenuUtils.setChecked(showgridMI,
            JaxoPrefs.getBooleanPref(JaxoPrefs.PREF_SHOWGRIDBAR));
        JaxoMenuUtils.setAccelerator(showgridMI, KeyEvent.VK_G, DEFAULT_MASK);
        JaxoMenuUtils.setActionCommand(showgridMI, JaxoConstants.GRIDBAR);
        JaxoMenuUtils.setLanguageKey(showgridMI, "Show_Gridbar");
        return showgridMI;
    }

    private JMenuItem showstatusMenuItem() {
        JaxoMenuUtils.setChecked(showStatusMI,
            JaxoPrefs.getBooleanPref(JaxoPrefs.PREF_SHOWSTATUS));
        JaxoMenuUtils.setAccelerator(showStatusMI, KeyEvent.VK_F, DEFAULT_MASK);
        JaxoMenuUtils.setActionCommand(showStatusMI, JaxoConstants.STATUSBAR);
        JaxoMenuUtils.setLanguageKey(showStatusMI, "Show_Statusbar");
        return showStatusMI;
    }

    private JMenuItem showtoolMenuItem() {
        JaxoMenuUtils.setChecked(showtoolMI,
            JaxoPrefs.getBooleanPref(JaxoPrefs.PREF_SHOWTOOL));
        JaxoMenuUtils.setAccelerator(showtoolMI, KeyEvent.VK_B, DEFAULT_MASK);
        JaxoMenuUtils.setActionCommand(showtoolMI, JaxoConstants.TOOLBAR);
        JaxoMenuUtils.setLanguageKey(showtoolMI, "Show_Toolbar");
        return showtoolMI;
    }

    private JaxoMenu vertexMenu() {
        final JaxoMenu vertexM = new VertexMenu(JaxoConstants.VERTEXT1);
        this.addPropertyChangeListener(vertexM);
        return vertexM;
    }

    private JaxoMenu defaultMenu() {
        final JaxoMenu defaultM = new DefaultMenu(JaxoPrefs.getIntPref(JaxoPrefs.PREF_DEFAULTACTION));
        this.addPropertyChangeListener(defaultM);
        return defaultM;
    }

    private JaxoMenu languageMenu() {
        final JaxoMenu languageM = new LanguageMenu(JaxoPrefs.getStringPref(JaxoPrefs.PREF_LANGUAGE));
        this.addPropertyChangeListener(languageM);
        return languageM;
    }

    private JMenuItem lookAndFeelMenuItem() {
        final JMenuItem lookAndFeelMI = new JMenuItem();
        JaxoMenuUtils.setAccelerator(lookAndFeelMI, KeyEvent.VK_L, DEFAULT_MASK);
        JaxoMenuUtils.setActionCommand(lookAndFeelMI, JaxoConstants.LOOKNFEEL);
        JaxoMenuUtils.setLanguageKey(lookAndFeelMI, "Look_And_Feel");
        return lookAndFeelMI;
    }

    private JMenuItem packageMenuItem() {
        final JMenuItem packageMI = new JMenuItem();
        JaxoMenuUtils.setAccelerator(packageMI, KeyEvent.VK_J, DEFAULT_MASK);
        JaxoMenuUtils.setActionCommand(packageMI, JaxoConstants.PACKAGE);
        JaxoMenuUtils.setLanguageKey(packageMI, "Add_Latex_package");
        return packageMI;
    }

    private JMenuItem describeMenuItem() {
        final JMenuItem describeMI = new JMenuItem();
        JaxoMenuUtils.setAccelerator(describeMI, KeyEvent.VK_D, DEFAULT_MASK);
        JaxoMenuUtils.setActionCommand(describeMI, JaxoConstants.DESCRIBE);
        JaxoMenuUtils.setLanguageKey(describeMI, "Add_description");
        return describeMI;
    }

    private JMenuItem moveGraphItem() {
        final JMenuItem result = new JMenuItem();
        JaxoMenuUtils.setActionCommand(result, JaxoConstants.MOVE_GRAPH);
        JaxoMenuUtils.setLanguageKey(result, "Move_graph");
        return result;
    }

}
