/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.gui.menu.popup;

import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;

import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.KeyStroke;

import net.sf.jaxodraw.util.JaxoConstants;
import net.sf.jaxodraw.util.JaxoLanguage;
import net.sf.jaxodraw.util.JaxoLocalized;
import net.sf.jaxodraw.util.JaxoLooknFeel;

/** A class that displays a pop-up menu if the user right-clicks on a tab.
 * @since 2.0
 */
public class JaxoCanvasPopupMenu extends JPopupMenu implements JaxoLocalized {
    private static final long serialVersionUID = 7526471155622776147L;
    private final JMenuItem newTabMI;
    private final JMenuItem renameMI;
    private final JMenuItem refreshMI;
    private final JMenuItem clearMI;
    private final JMenuItem saveMI;
    private final JMenuItem saveAsMI;
    private final JMenuItem exportMI;
    private final JMenuItem closeTabMI;
    private final JMenuItem pasteMI;

    /**
     * Constructor: sets all the menu entries and the action listeners.
     *
     * @param listener an ActionListener to receive events from this PopupMenu.
     */
    public JaxoCanvasPopupMenu(final ActionListener listener) {
        super();
        final int keyMask = Toolkit.getDefaultToolkit().getMenuShortcutKeyMask();

        newTabMI = new JMenuItem();
        newTabMI.addActionListener(listener);
        newTabMI.setActionCommand(JaxoConstants.getModeAsString(
                JaxoConstants.NEWT));
        newTabMI.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_T, keyMask));

        renameMI = new JMenuItem();
        renameMI.addActionListener(listener);
        renameMI.setActionCommand(JaxoConstants.getModeAsString(
                JaxoConstants.RENAME_TAB));

        refreshMI = new JMenuItem();
        refreshMI.addActionListener(listener);
        refreshMI.setActionCommand(JaxoConstants.getModeAsString(
                JaxoConstants.REFRESH));
        refreshMI.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_S,
                ActionEvent.ALT_MASK));

        clearMI = new JMenuItem();
        clearMI.addActionListener(listener);
        clearMI.setActionCommand(JaxoConstants.getModeAsString(
                JaxoConstants.CLEAR));
        clearMI.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_X,
                ActionEvent.ALT_MASK));

        pasteMI = new JMenuItem();
        pasteMI.addActionListener(listener);
        pasteMI.setActionCommand(JaxoConstants.getModeAsString(
                JaxoConstants.PASTE));
        pasteMI.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_V,
                ActionEvent.CTRL_MASK));

        saveMI = new JMenuItem();
        saveMI.addActionListener(listener);
        saveMI.setActionCommand(JaxoConstants.getModeAsString(
                JaxoConstants.SAVE));

        saveAsMI = new JMenuItem();
        saveAsMI.addActionListener(listener);
        saveAsMI.setActionCommand(JaxoConstants.getModeAsString(
                JaxoConstants.SAVE_AS));

        exportMI = new JMenuItem();
        exportMI.addActionListener(listener);
        exportMI.setActionCommand(JaxoConstants.getModeAsString(
                JaxoConstants.EXPORT));

        closeTabMI = new JMenuItem();
        closeTabMI.addActionListener(listener);
        closeTabMI.setActionCommand(JaxoConstants.getModeAsString(
                JaxoConstants.CLOSE));
        closeTabMI.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_X,
                keyMask | ActionEvent.SHIFT_MASK));

        this.add(saveMI);
        this.add(saveAsMI);
        this.add(exportMI);
        this.add(renameMI);
        this.addSeparator();
        this.add(clearMI);
        this.add(pasteMI);
        this.addSeparator();
        this.add(refreshMI);
        this.addSeparator();
        this.add(closeTabMI);
        this.addSeparator();
        this.add(newTabMI);

        updateLanguage();

        JaxoLooknFeel.registerComponent(this);
    }

    /** {@inheritDoc} */
    public final void updateLanguage() {
        saveMI.setText(JaxoLanguage.translate("Save"));
        saveAsMI.setText(JaxoLanguage.translate("Save_as"));
        exportMI.setText(JaxoLanguage.translate("Export"));
        newTabMI.setText(JaxoLanguage.translate("New_tab"));
        clearMI.setText(JaxoLanguage.translate("Clear"));
        refreshMI.setText(JaxoLanguage.translate("Refresh"));
        renameMI.setText(JaxoLanguage.translate("Rename_tab"));
        pasteMI.setText(JaxoLanguage.translate("Paste"));
        closeTabMI.setText(JaxoLanguage.translate("Close"));
    }
}
