/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.gui.menu.popup;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;

import javax.swing.ButtonGroup;
import javax.swing.JPopupMenu;
import javax.swing.JRadioButtonMenuItem;
import javax.swing.KeyStroke;

import net.sf.jaxodraw.util.JaxoConstants;
import net.sf.jaxodraw.util.JaxoLanguage;
import net.sf.jaxodraw.util.JaxoLocalized;
import net.sf.jaxodraw.util.JaxoLooknFeel;
import net.sf.jaxodraw.util.JaxoUtils;


/** Pops up a menu with the dufferent vertex types whenever the user
 * right clicks on the vertex button.
 * @since 2.0
 */
public class JaxoVertexPopupMenu extends JPopupMenu implements JaxoLocalized {
    private static final long serialVersionUID = 7526471155622776147L;
    private final JRadioButtonMenuItem type1BT;
    private final JRadioButtonMenuItem type2BT;
    private final JRadioButtonMenuItem type3BT;
    private final JRadioButtonMenuItem type4BT;
    private final JRadioButtonMenuItem type5BT;
    private final JRadioButtonMenuItem type6BT;

    /** Constructs a new vertex popup menu. */
    public JaxoVertexPopupMenu() {
        super();
        final ButtonGroup vertexGroup = new ButtonGroup();
        type1BT =
            new JRadioButtonMenuItem(JaxoUtils.newImageIcon("vertexsT1.png"));
        type1BT.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_D,
                ActionEvent.SHIFT_MASK));
        type2BT =
            new JRadioButtonMenuItem(JaxoUtils.newImageIcon("vertexsT2.png"));
        type2BT.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_C,
                ActionEvent.SHIFT_MASK));
        type3BT =
            new JRadioButtonMenuItem(JaxoUtils.newImageIcon("vertexsT3.png"));
        type3BT.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_S,
                ActionEvent.SHIFT_MASK));
        type4BT =
            new JRadioButtonMenuItem(JaxoUtils.newImageIcon("vertexsT4.png"));
        type4BT.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_X,
                ActionEvent.SHIFT_MASK));
        type5BT =
            new JRadioButtonMenuItem(JaxoUtils.newImageIcon("vertexsT5.png"));
        type5BT.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_T,
                ActionEvent.SHIFT_MASK));
        type6BT =
            new JRadioButtonMenuItem(JaxoUtils.newImageIcon("vertexsT6.png"));
        type6BT.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_I,
                ActionEvent.SHIFT_MASK));

        type1BT.setActionCommand(JaxoConstants.getModeAsString(
                JaxoConstants.VERTEXT1));
        type1BT.setSelected(true);
        type2BT.setActionCommand(JaxoConstants.getModeAsString(
                JaxoConstants.VERTEXT2));
        type3BT.setActionCommand(JaxoConstants.getModeAsString(
                JaxoConstants.VERTEXT3));
        type4BT.setActionCommand(JaxoConstants.getModeAsString(
                JaxoConstants.VERTEXT4));
        type5BT.setActionCommand(JaxoConstants.getModeAsString(
                JaxoConstants.VERTEXT5));
        type6BT.setActionCommand(JaxoConstants.getModeAsString(
                JaxoConstants.VERTEXT6));

        vertexGroup.add(type1BT);
        vertexGroup.add(type2BT);
        vertexGroup.add(type3BT);
        vertexGroup.add(type4BT);
        vertexGroup.add(type5BT);
        vertexGroup.add(type6BT);

        this.add(type1BT);
        this.add(type2BT);
        this.add(type3BT);
        this.add(type4BT);
        this.add(type5BT);
        this.add(type6BT);

        updateLanguage();

        JaxoLooknFeel.registerComponent(this);
    }

    /** Adds an action listener to this popup menu.
    * @param l The ActionListener to add.
    */
    public void addActionListener(final ActionListener l) {
        type1BT.addActionListener(l);
        type2BT.addActionListener(l);
        type3BT.addActionListener(l);
        type4BT.addActionListener(l);
        type5BT.addActionListener(l);
        type6BT.addActionListener(l);
    }

    /** {@inheritDoc} */
    public final void updateLanguage() {
        type1BT.setText(JaxoLanguage.translate("JaxoVertex.type.dot"));
        type2BT.setText(JaxoLanguage.translate("JaxoVertex.type.circleCross"));
        type3BT.setText(JaxoLanguage.translate("JaxoVertex.type.square"));
        type4BT.setText(JaxoLanguage.translate("JaxoVertex.type.cross"));
        type5BT.setText(JaxoLanguage.translate("JaxoVertex.type.triangle"));
        type6BT.setText(JaxoLanguage.translate("JaxoVertex.type.diamond"));
    }

    /** Notification on change of vertex type.
     * Sets the radio button to the current vertex type in the pop-up menu.
     * @param value Specifies the current vertex type.
     */
    public final void updateVertexType(final int value) {
        if (value == JaxoConstants.VERTEXT1) {
            type1BT.setSelected(true);
        } else if (value == JaxoConstants.VERTEXT2) {
            type2BT.setSelected(true);
        } else if (value == JaxoConstants.VERTEXT3) {
            type3BT.setSelected(true);
        } else if (value == JaxoConstants.VERTEXT4) {
            type4BT.setSelected(true);
        } else if (value == JaxoConstants.VERTEXT5) {
            type5BT.setSelected(true);
        } else if (value == JaxoConstants.VERTEXT6) {
            type6BT.setSelected(true);
        }
    }
}
