/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.gui.menu.popup;

import java.awt.event.ActionListener;

import javax.swing.ButtonGroup;
import javax.swing.JMenu;
import javax.swing.JPopupMenu;
import javax.swing.JRadioButtonMenuItem;

import net.sf.jaxodraw.util.JaxoConstants;
import net.sf.jaxodraw.util.JaxoLanguage;
import net.sf.jaxodraw.util.JaxoLocalized;
import net.sf.jaxodraw.util.JaxoLooknFeel;


/** A class that displays a pop-up menu if the user right-clicks on a tab.
 * @since 2.0
 */
public class JaxoZoomPopupMenu extends JPopupMenu implements JaxoLocalized {
    private static final long serialVersionUID = 7526471155622776147L;
    private final JMenu zoomFactorM;
    private final JRadioButtonMenuItem zoom2;
    private final JRadioButtonMenuItem zoom4;
    private final JRadioButtonMenuItem zoom8;

    /** Constructor: sets all the menu entries. */
    public JaxoZoomPopupMenu() {
        super();
        zoomFactorM = new JMenu();
        zoomFactorM.setActionCommand(JaxoConstants.getModeAsString(
                JaxoConstants.ZOOM_FACTOR));

        final ButtonGroup zoomFactorGroup = new ButtonGroup();

        zoom2 = new JRadioButtonMenuItem();
        zoom2.setActionCommand(JaxoConstants.getModeAsString(
                JaxoConstants.ZOOM_FACTOR_X2));
        zoom4 = new JRadioButtonMenuItem();
        zoom4.setActionCommand(JaxoConstants.getModeAsString(
                JaxoConstants.ZOOM_FACTOR_X4));
        zoom8 = new JRadioButtonMenuItem();
        zoom8.setActionCommand(JaxoConstants.getModeAsString(
                JaxoConstants.ZOOM_FACTOR_X8));

        zoomFactorGroup.add(zoom2);
        zoomFactorGroup.add(zoom4);
        zoomFactorGroup.add(zoom8);
        zoomFactorM.add(zoom2);
        zoomFactorM.add(zoom4);
        zoomFactorM.add(zoom8);

        zoom2.setSelected(true);

        this.add(zoomFactorM);

        updateLanguage();

        JaxoLooknFeel.registerComponent(this);
    }

    /** Sets the zoom factor to the given value.
     * @param value The new zoom factor.
     */
    public void setZoomFactor(final int value) {
        if (value <= 2) {
            zoom2.setSelected(true);
        } else if (value <= 4) {
            zoom4.setSelected(true);
        } else {
            zoom8.setSelected(true);
        }
    }

    /** Adds an action listener to this popup menu.
    * @param l The ActionListener to add.
    */
    public void addActionListener(final ActionListener l) {
        zoomFactorM.addActionListener(l);
        zoom2.addActionListener(l);
        zoom4.addActionListener(l);
        zoom8.addActionListener(l);
    }

    /** {@inheritDoc} */
    public final void updateLanguage() {
        zoomFactorM.setText(JaxoLanguage.translate("JaxoZoom.factor"));

        zoom2.setText(JaxoLanguage.message("JaxoZoom.factor.%0", Integer.valueOf(2)));
        zoom4.setText(JaxoLanguage.message("JaxoZoom.factor.%0", Integer.valueOf(4)));
        zoom8.setText(JaxoLanguage.message("JaxoZoom.factor.%0", Integer.valueOf(8)));
    }
}
