/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.gui.panel;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Point;

import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;

import javax.swing.BorderFactory;
import javax.swing.JLabel;
import javax.swing.JWindow;

import net.sf.jaxodraw.graph.JaxoGraph;
import net.sf.jaxodraw.object.JaxoObject;
import net.sf.jaxodraw.object.text.JaxoLatexText;
import net.sf.jaxodraw.util.JaxoDictionary;


/** Show tooltips if the mouse moves over JaxoLatexText objects
 * on the JaxoCanvas in the focused window.
 * Note: For simplicity, this implements listener methods directly.
 * Do not call or override.
 * @since 2.0
 */
public class JaxoLatexTextToolTip extends MouseAdapter {
    private final JaxoDictionary language = new JaxoDictionary(JaxoLatexTextToolTip.class);

    private final JWindow window;
    private final JLabel label;

    private boolean isActive = true;
    private JaxoGraph theGraph;

    /**
     * Constructor.
     */
    public JaxoLatexTextToolTip() {
        super();
        label = new JLabel();
        label.setBorder(BorderFactory.createLineBorder(new Color(0, 0, 0)));
        label.setBackground(new Color(250, 250, 200));
        label.setOpaque(true);

        window = new JWindow();
        window.setContentPane(label);
    }

    /**
     * Sets the graph.
     *
     * @param graph the graph. May be null.
     */
    public void setGraph(final JaxoGraph graph) {
        this.theGraph = graph;
    }

    /**
     * Activates this JaxoLatexTextToolTip. By default, the ToolTip is activated.
     *
     * @param active true to activate the ToolTip, false otherwise.
     */
    public void setActive(final boolean active) {
        this.isActive = active;

        if (!isActive) {
            window.setVisible(false);
        }
    }

    /**
     * Process the event when the mouse enters the canvas.
     * Perform action, just in case.
     *
     * @param e The mouse event to process.
     */
    @Override
    public final void mouseEntered(final MouseEvent e) {
        mouseMoved(e);
    }

    /**
     * The action to be taken when the mouse is moved on the canvas.
     * Checks for Latex texts and shows the latex panel.
     * @param e The mouse event.
     */
    @Override
    public final void mouseMoved(final MouseEvent e) {
        if (!isActive || theGraph == null) {
            return;
        }

        // Note: It is assumed that this graph shares the coordinate system
        // with the event source (canvas), e.g., no Insets on Canvas
        for (int i = 0; i < theGraph.listSize(); i++) {
            final JaxoObject o = theGraph.listElementAt(i);

            if (o instanceof JaxoLatexText) {
                final JaxoLatexText text = (JaxoLatexText) o;
                if (text.isRolledOver(e.getX(), e.getY())) {
                    label.setText(language.message("texString%0", text.getTextString()));

                    final Point p = e.getComponent().getLocationOnScreen();

                    p.x += e.getX();
                    p.y += e.getY();

                    // offset Window somewhat (also necessary if a separate Window
                    // is used - if the mouse enters the Window, it is lost here)
                    final Dimension d = window.getPreferredSize();

                    p.x -= (d.width / 3);
                    p.y -= (d.height * 4);

                    window.setBounds(p.x, p.y, d.width, d.height);
                    window.setVisible(true);
                    return;
                }
            }
        }

        window.setVisible(false);
    }

    /**
     * Process the event when the mouse exits the canvas. Hide window.
     *
     * @param e The mouse event to process.
     */
    @Override
    public final void mouseExited(final MouseEvent e) {
        window.setVisible(false);
    }

    /**
     * Hide window when the mouse is pressed on the canvas.
     *
     * @param e The mouse event to process.
     */
    @Override
    public final void mousePressed(final MouseEvent e) {
        window.setVisible(false);
    }
}
