/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.gui.panel.button;

import java.awt.Color;
import java.awt.event.ActionEvent;

import javax.swing.JButton;

import net.sf.jaxodraw.gui.panel.JaxoColorChooser;
import net.sf.jaxodraw.util.JaxoUtils;
import net.sf.jaxodraw.util.JaxoDictionary;
import net.sf.jaxodraw.util.Location;


/**
 * Button displaying a color selection - typically used in such a way
 * that clicking it opens a color chooser to change the color.
 * @since 2.0
 */
public class JaxoColorButton extends JButton {
    private static final JaxoDictionary LANGUAGE = new JaxoDictionary(JaxoColorButton.class);
    private static final Color VOID_COLOR = new Color(0, true);
    private static final long serialVersionUID = 7526471155622776147L;
    private Color color;
    private int chooserMode;
    private String chooserTitle;
    private JaxoColorChooser chooser;
    private boolean optional;

    /** with Color.black. */
    public JaxoColorButton() {
        this(Color.black);
    }

    /**
     * with the given color.
     * @param c The color of the button.
     */
    public JaxoColorButton(final Color c) {
        super();
        if (c == null) {
            optional = true;
        }
        color = c;

        setVerticalTextPosition(CENTER);
        setHorizontalTextPosition(CENTER);

        updateIconText();

        //chooserMode = JaxoColorChooser.MODE_JAXO_COLORS;
    }

    private void updateIconText() {
        setIcon(JaxoUtils.getChooserImageIcon(color == null ? VOID_COLOR : color));
        setText(color == null ? LANGUAGE.value("noColor") : " ");
    }

    /**
     * Current optional property.
     * @return The current optional value.
     */
    public final boolean isOptional() {
        return optional;
    }

    /**
     * Sets the optional property.
     * @param value The value to set.
     */
    public void setOptional(final boolean value) {
        optional = value;
    }

    /**
     * Current color property.
     * @return The current color.
     */
    public final Color getColor() {
        return color;
    }

    /**
     * Sets the button color.
     * @param value The color to set.
     */
    public void setColor(final Color value) {
        if (value == null ? color != null : !value.equals(color)) {
            final Color old = color;

            color = value;
            updateIconText();
            repaint();

            firePropertyChange("color", old, color);
        }
    }

    /**
     * Returns the current chooser mode.
     * @see JaxoColorChooser#getMode()
     * @return The chooser mode.
     */
    public final int getChooserMode() {
        return chooserMode;
    }

    /**
     * Sets the current chooser mode.
     * @param value The mode to set.
     */
    public void setChooserMode(final int value) {
        chooserMode = value;
    }

    /**
     * The title of the dialog.
     * @see JaxoColorChooser#getDialogTitle()
     * @return The title.
     */
    public final String getChooserTitle() {
        return chooserTitle;
    }

    /**
     * Sets the current chooser title.
     * @param value The title.
     */
    public void setChooserTitle(final String value) {
        chooserTitle = value;
    }

    /**
     * Chooser to be shown to change the current color automatically
     * on ActionEvents. The default is 'null'.
     * @return The chooser.
     */
    public final JaxoColorChooser getChooser() {
        return chooser;
    }

    /**
     * Sets the chooser.
     * @param value The chooser to set.
     */
    public void setChooser(final JaxoColorChooser value) {
        chooser = value;
    }


    /** Show the (existing) chooser programmatically, as if the
        button were clicked.
    */
    public void showChooser() {
        if (chooser != null) {
            chooser.setMode(chooserMode);
            chooser.setDialogTitle(chooserTitle);
            chooser.setColor(getColor());
            chooser.setOptional(optional);
            chooser.show(new Location.RelativeTo(this));
            setColor(chooser.getColor());
        }
    }

    /**
     * Apply an action event.
     *
     * @param e the action event.
     */
    @Override
    protected void fireActionPerformed(final ActionEvent e) {
        super.fireActionPerformed(e);
        showChooser();
    }
}
