/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.gui.panel.button;

import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.event.ActionListener;

import javax.swing.JPanel;

import net.sf.jaxodraw.util.JaxoConstants;
import net.sf.jaxodraw.util.JaxoLanguage;
import net.sf.jaxodraw.util.JaxoLocalized;
import net.sf.jaxodraw.util.JaxoUtils;


/** Responsible for displaying and modifying the buttons of the
 * edit button menu.
 * @since 2.0
 */
public class JaxoEditButtons extends JPanel implements JaxoLocalized {
    private static final long serialVersionUID = 7526471155622776147L;
    private static final String[] EDIT_BUTTON_ICONS =
    {
        "move.png", "resize.png", "color.png", "edit.png", "duplicate.png",
        "back.png", "fore.png", "delete.png", "group.png", "ungroup.png",
        "empty.png", "empty.png"
    };
    private static final int[] EDIT_MODES =
    {
        JaxoConstants.MOVE, JaxoConstants.RESIZE, JaxoConstants.COLOR,
        JaxoConstants.EDIT, JaxoConstants.DUPLICATE, JaxoConstants.BACKGROUND,
        JaxoConstants.FOREGROUND, JaxoConstants.DELETE, JaxoConstants.SELECT,
        JaxoConstants.UNGROUP, JaxoConstants.DUMMIE, JaxoConstants.DUMMIE1
    };
    private static final int NOF_BUTTONS = EDIT_MODES.length;
    private static final String[] TOOLTIP_KEYS =
    {
        "Move_an_object", "Resize_an_object", "Color_an_object",
        "Edit_an_object", "Duplicate_an_object",
        "Put_an_object_in_the_background", "Put_an_object_in_the_foreground",
        "Delete_an_object", "Select_objects", "Ungroup_objects", "", ""
    };
    private final JaxoPanelButton[] editButtons =
        new JaxoPanelButton[NOF_BUTTONS];

    /** Constructor: lays out the edit button panel. */
    public JaxoEditButtons() {
        super(new GridBagLayout());

        for (int i = 0; i < NOF_BUTTONS; i++) {
            editButtons[i] = new JaxoPanelButton();
            editButtons[i].setIcon(JaxoUtils.newImageIcon(EDIT_BUTTON_ICONS[i]));
            editButtons[i].setToolTipText(JaxoLanguage.translate(
                    TOOLTIP_KEYS[i]));
            editButtons[i].setId(JaxoConstants.getModeAsString(EDIT_MODES[i]));
        }

        final GridBagConstraints c =
            (GridBagConstraints) JaxoButtonPanel.BUTTON_CONSTRAINTS.clone();

        for (int y = 0; y < (NOF_BUTTONS / 4); y++) {
            for (int x = 0; x < 4; x++) {
                c.gridx = x;
                c.gridy = y;
                add(editButtons[x + (4 * y)], c);
            }
        }

        editButtons[10].setEnabled(false);
        editButtons[11].setEnabled(false);

    }

    /** Adds the given ActionListener to all buttons in this panel.
     * @param l The ActionListener to add.
     */
    public final void addActionListener(final ActionListener l) {
        for (int i = 0; i < NOF_BUTTONS; i++) {
            editButtons[i].addActionListener(l);
        }
    }

    /** Sets all the buttons as not pressed. */
    private void unsetButtons() {
        for (int i = 0; i < NOF_BUTTONS; i++) {
            editButtons[i].setSelected(false);
        }
    }

    /** Returns the index of the edit button corresponding to the edit event i.
     * @param i An edit event as defined in {@link JaxoConstants JaxoConstants}.
     * @return The index of the edit button corresponding to the edit event i.
     */
    private int getIndex(final int i) {
        int j = NOF_BUTTONS + 1;

        for (int jj = 0; jj < NOF_BUTTONS; jj++) {
            if (i == EDIT_MODES[jj]) {
                j = jj;
            }
        }

        return j;
    }

    /** Sets the specified button as pressed.
     * If mode is not an edit mode, all buttons are set as not pressed.
     * @param mode The mode specifying the button to be pressed,
     * as defined in {@link JaxoConstants JaxoConstants}.
     */
    public final void pressButton(final int mode) {
        unsetButtons();
        if (JaxoConstants.isEditMode(mode)) {
            final int j = getIndex(mode);
            editButtons[j].setSelected(true);
        }
    }

    /** Set the button corresponding to mode as default.
     * If mode is not an edit mode, all buttons are set as not default.
     * @param mode The button to set as default.
     */
    public final void setDefault(final int mode) {
        resetDefault();
        if (JaxoConstants.isEditMode(mode)) {
            final int j = getIndex(mode);
            editButtons[j].setDefault(true);
        }
    }

    /** Reset default to false for all buttons. */
    private void resetDefault() {
        for (int i = 0; i < NOF_BUTTONS; i++) {
            editButtons[i].setDefault(false);
        }
    }

    /** {@inheritDoc} */
    public final void updateLanguage() {
        for (int i = 0; i < NOF_BUTTONS; i++) {
            editButtons[i].setToolTipText(JaxoLanguage.translate(
                    TOOLTIP_KEYS[i]));
        }
    }
}
