/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.gui.panel.button;

import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.event.ActionListener;

import javax.swing.JPanel;

import net.sf.jaxodraw.util.JaxoConstants;
import net.sf.jaxodraw.util.JaxoLanguage;
import net.sf.jaxodraw.util.JaxoLocalized;
import net.sf.jaxodraw.util.JaxoUtils;


/** Responsible for displaying and modifying the buttons of the
 * particle button menu.
 * @since 2.0
 */
public class JaxoParticleButtons extends JPanel implements JaxoLocalized {
    private static final long serialVersionUID = 7526471155622776147L;
    private static final String[] PARTICLE_BUTTON_ICONS =
    {
        "fline.png", "floop.png", "farc.png", "bezierf.png", "sline.png",
        "sloop.png", "sarc.png", "beziers.png", "gline.png", "gloop.png",
        "garc.png", "bezierg.png", "photon.png", "photonloop.png",
        "photonarc.png", "bezierph.png", "gluon.png", "gluonloop.png",
        "gluonarc.png", "beziergl.png"
    };
    private static final int[] PARTICLE_MODES =
    {
        JaxoConstants.FLINE, JaxoConstants.FLOOP, JaxoConstants.FARC,
        JaxoConstants.FBEZ, JaxoConstants.SLINE, JaxoConstants.SLOOP,
        JaxoConstants.SARC, JaxoConstants.SBEZ, JaxoConstants.GLINE,
        JaxoConstants.GLOOP, JaxoConstants.GARC, JaxoConstants.GBEZ,
        JaxoConstants.PLINE, JaxoConstants.PLOOP, JaxoConstants.PARC,
        JaxoConstants.PBEZ, JaxoConstants.GLLINE, JaxoConstants.GLLOOP,
        JaxoConstants.GLARC, JaxoConstants.GLBEZ
    };
    private static final int NOF_BUTTONS = PARTICLE_MODES.length;
    private static final String[] TOOLTIP_KEYS =
    {
        "Draw_a_fermion_line", "Draw_a_fermion_loop", "Draw_a_fermion_arc",
        "Draw_a_fermion_bezier", "Draw_a_scalar_line", "Draw_a_scalar_loop",
        "Draw_a_scalar_arc", "Draw_a_scalar_bezier", "Draw_a_ghost_line",
        "Draw_a_ghost_loop", "Draw_a_ghost_arc", "Draw_a_ghost_bezier",
        "Draw_a_photon_line", "Draw_a_photon_loop", "Draw_a_photon_arc",
        "Draw_a_photon_bezier", "Draw_a_gluon_line", "Draw_a_gluon_loop",
        "Draw_a_gluon_arc", "Draw_a_gluon_bezier"
    };
    private final JaxoPanelButton[] particleButtons =
        new JaxoPanelButton[NOF_BUTTONS];

    /** Constructor: lays out the particle button panel. */
    public JaxoParticleButtons() {
        super(new GridBagLayout());

        for (int i = 0; i < NOF_BUTTONS; i++) {
            particleButtons[i] = new JaxoPanelButton();
            particleButtons[i].setIcon(JaxoUtils.newImageIcon(
                    PARTICLE_BUTTON_ICONS[i]));
            particleButtons[i].setToolTipText(JaxoLanguage.translate(
                    TOOLTIP_KEYS[i]));
            particleButtons[i].setId(JaxoConstants.getModeAsString(
                    PARTICLE_MODES[i]));
        }

        final GridBagConstraints c =
            (GridBagConstraints) JaxoButtonPanel.BUTTON_CONSTRAINTS.clone();

        for (int y = 0; y < 5; y++) {
            for (int x = 0; x < 4; x++) {
                c.gridx = x;
                c.gridy = y;
                add(particleButtons[x + (4 * y)], c);
            }
        }
    }

    /** Adds the given ActionListener to all buttons in this panel.
     * @param l The ActionListener to add.
     */
    public final void addActionListener(final ActionListener l) {
        for (int i = 0; i < NOF_BUTTONS; i++) {
            particleButtons[i].addActionListener(l);
        }
    }

    /** Sets all the buttons as not pressed. */
    private void unsetButtons() {
        for (int i = 0; i < NOF_BUTTONS; i++) {
            particleButtons[i].setSelected(false);
        }
    }

    /** Returns the index of the particle button corresponding
     * to the particle event i.
     * @param i A particle event as defined in
     * {@link JaxoConstants JaxoConstants}.
     * @return The index of the particle button corresponding
     * to the particle event i.
     */
    private int getIndex(final int i) {
        int j = NOF_BUTTONS + 1;

        for (int jj = 0; jj < NOF_BUTTONS; jj++) {
            if (i == PARTICLE_MODES[jj]) {
                j = jj;
            }
        }

        return j;
    }

    /** Set the button corresponding to mode as default.
     * If mode is not a particle mode, all buttons are set as not default.
     * @param mode The button to set as default.
     */
    public final void setDefault(final int mode) {
        resetDefault();
        if (JaxoConstants.isParticleMode(mode)) {
            final int j = getIndex(mode);
            particleButtons[j].setDefault(true);
        }
    }

    /** Reset default to false for all buttons. */
    private void resetDefault() {
        for (int i = 0; i < NOF_BUTTONS; i++) {
            particleButtons[i].setDefault(false);
        }
    }

    /** Sets the specified button as pressed.
     * If mode is not a particle mode, all buttons are set as not pressed.
     * @param mode The mode specifying the button to be pressed,
     * as defined in {@link JaxoConstants JaxoConstants}.
     */
    public final void pressButton(final int mode) {
        unsetButtons();
        if (JaxoConstants.isParticleMode(mode)) {
            final int j = getIndex(mode);
            particleButtons[j].setSelected(true);
        }
    }

    /** {@inheritDoc} */
    public final void updateLanguage() {
        for (int i = 0; i < NOF_BUTTONS; i++) {
            particleButtons[i].setToolTipText(JaxoLanguage.translate(
                    TOOLTIP_KEYS[i]));
        }
    }
}
