/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.gui.panel.edit;

import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;

import javax.swing.JCheckBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JSpinner;
import javax.swing.border.TitledBorder;

import net.sf.jaxodraw.util.JaxoLanguage;


/** A panel to change the arrow of an object.
 * @since 2.0
 */
public class JaxoArrowEditPanel extends JaxoEditPanel {
    private static final long serialVersionUID = 7526471155622776147L;

    private final transient JCheckBox arrowcb;
    private final transient JCheckBox flipcb;
    private final transient JSpinner spstin;

    /** Constructor: an arrow edit panel with arrow not flipped. */
    public JaxoArrowEditPanel() {
        this(true, false, 0.5f);
    }

     /** Constructor: returns a arrow edit panel with the given params.
     * @param arrow Whether the object has an arrow.
     * @param flip  Whether the arrow is flipped.
     * @param oldPosition The old position of the arrow
     */
    public JaxoArrowEditPanel(final boolean arrow, final boolean flip, final float oldPosition) {
        super();
        this.arrowcb =
            new JCheckBox(JaxoLanguage.translate("JaxoArrowEditPanel.arrow"));
        this.flipcb =
            new JCheckBox(JaxoLanguage.translate("JaxoArrowEditPanel.flip"));
        this.spstin = getSpinner(oldPosition, 0.f, 1.0f, 0.005f, "arrowpos");

        init(arrow, flip);
    }

    private void init(final boolean arrow, final boolean flip) {
        final JLabel arpLabel = createI18NLabel("JaxoArrowPositionEditPanel.position");

        arrowcb.setName("arrow");
        arrowcb.setSelected(arrow);
        arrowcb.addItemListener(new ItemListener() {
                public final void itemStateChanged(final ItemEvent e) {
                    flipcb.setEnabled(arrowcb.isSelected());
                    spstin.setEnabled(arrowcb.isSelected());
                    arpLabel.setEnabled(arrowcb.isSelected());
                }
            });

        final JPanel arpPanel = new JPanel();
        //hard code the preferred spinner dimension (look necessary here)
        spstin.setPreferredSize(new Dimension(55, 20));
        arpPanel.add(arpLabel);
        arpPanel.add(spstin);

        flipcb.setName("flip");
        flipcb.setSelected(flip);
        flipcb.setEnabled(arrow);


        final JPanel arrEPanel = new JPanel();
        final GridBagLayout gb = new GridBagLayout();
        final GridBagConstraints sc = new GridBagConstraints();
        arrEPanel.setLayout(gb);

        sc.anchor = GridBagConstraints.WEST;

        sc.gridx = 0;
        sc.gridy = 0;
        arrEPanel.add(arrowcb, sc);

        sc.gridx = 1;
        sc.gridy = 0;
        arrEPanel.add(flipcb, sc);

        sc.gridx = 0;
        sc.gridy = 1;
        sc.gridwidth = 2;
        arrEPanel.add(arpPanel, sc);

        setLineBoxLayout();

        final TitledBorder arrETitle = createI18NBorder("JaxoArrowEditPanel.title");
        this.setBorder(arrETitle);
        this.add(arrEPanel);

    }

    /** {@inheritDoc} */
    public void addEditPanelListener(final JaxoEditPanelListener listener) {
        if (listener != null) {
            arrowcb.addItemListener(listener);
            flipcb.addItemListener(listener);
            spstin.addChangeListener(listener);
            //moreopt.addActionListener(listener);
        }
    }
}
