/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.gui.panel.edit;

import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;

import javax.swing.JCheckBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JSpinner;
import javax.swing.SpinnerNumberModel;
import javax.swing.border.TitledBorder;

import net.sf.jaxodraw.gui.swing.spinner.JaxoFixedJSpinner;
import net.sf.jaxodraw.util.JaxoLanguage;


/** A panel to change the double line property of an object.
 * @since 2.0
 */
public class JaxoDLEditPanel extends JaxoEditPanel {
    private static final long serialVersionUID = 7526471155622776147L;
    private transient JCheckBox dlcb;
    private transient JSpinner spsep;

    /** Constructor: a double line edit panel with default values. */
    public JaxoDLEditPanel() {
        this(false, 1.f);
    }

    /** Constructor: a double line edit panel with the given params.
     * @param dLine Whether the object is double lined.
     * @param dLSep The double line separation.
     */
    public JaxoDLEditPanel(final boolean dLine, final float dLSep) {
        super();
        dlcb =
            new JCheckBox(JaxoLanguage.translate("JaxoDLEditPanel.doubleLine"));
        dlcb.setName("dLine");
        dlcb.setSelected(dLine);
        dlcb.addItemListener(new ItemListener() {
                public final void itemStateChanged(final ItemEvent e) {
                    spsep.setEnabled(dlcb.isSelected());
                }
            });

        final SpinnerNumberModel modelsep =
            new SpinnerNumberModel(dLSep, 0.5, 10.0, 0.5);
        final JLabel stsep = createI18NLabel("JaxoDLEditPanel.separation");
        spsep = new JaxoFixedJSpinner(modelsep);
        spsep.setName("dlsep");
        spsep.setEnabled(dLine);

        final JPanel dlP = new JPanel();
        final GridBagLayout subdl = new GridBagLayout();
        final GridBagConstraints subcdl = new GridBagConstraints();
        dlP.setLayout(subdl);

        subcdl.gridx = 0;
        subcdl.gridy = 0;
        dlP.add(dlcb, subcdl);

        subcdl.gridx = 0;
        subcdl.gridy = 1;
        subcdl.insets = new Insets(5, 0, 5, 0);
        dlP.add(stsep, subcdl);

        subcdl.gridx = 1;
        subcdl.gridy = 1;
        subcdl.insets = new Insets(5, 0, 5, 5);
        spsep.setPreferredSize(new Dimension(50, 20));
        dlP.add(spsep, subcdl);

        setLineBoxLayout();

        final TitledBorder dlTitle = createI18NBorder("JaxoDLEditPanel.title");
        this.setBorder(dlTitle);
        this.add(dlP);
    }

    /** {@inheritDoc} */
    public void addEditPanelListener(final JaxoEditPanelListener listener) {
        if (listener != null) {
            dlcb.addItemListener(listener);
            spsep.addChangeListener(listener);
        }
    }
}
