/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.gui.panel.edit;

import java.awt.BorderLayout;
import java.awt.Font;
import java.awt.event.ItemListener;
import java.awt.event.ItemEvent;

import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JSpinner;

import net.sf.jaxodraw.gui.panel.JaxoPrefsPanel;
import net.sf.jaxodraw.gui.swing.JaxoFontStyleRenderer;
import net.sf.jaxodraw.gui.swing.spinner.JaxoSpinnerComboBox;
import net.sf.jaxodraw.object.text.JaxoPSText;
import net.sf.jaxodraw.util.JaxoLanguage;
import net.sf.jaxodraw.util.JaxoPrefs;


/** A panel to change the text of a text object.
 * @since 2.0
 */
public class JaxoPSFontPanel extends JaxoEditPanel {
    private static final long serialVersionUID = 7526471155622776147L;
    // private fields
    private transient JComboBox famcomb;
    private transient JComboBox sizecomb;
    private transient JComboBox stycomb;

    /** Constructor: returns a PSFont edit panel with
     * initial values taken from the preferred font.
     */
    public JaxoPSFontPanel() {
        this(new Font(JaxoPrefs.getStringPref(JaxoPrefs.PREF_PSFAMILY),
                JaxoPrefs.getIntPref(JaxoPrefs.PREF_PSSTYLE),
                JaxoPrefs.getIntPref(JaxoPrefs.PREF_PSSIZE)));
    }

    /** Constructor: returns a  PSFont edit panel with
     * initial values taken from the given font.
     * @param font The initial font.
     */
    public JaxoPSFontPanel(final Font font) {
        super();
        this.famcomb =
            new JComboBox(JaxoPrefsPanel.getSystemFonts().toArray());
        famcomb.setName("psfontname");
        int fontIndex;
        // note: for groups, font may be null, set to no selection
        if (font == null) {
            fontIndex = -1;
        } else {
            fontIndex = JaxoPrefsPanel.getSystemFonts().indexOf(font.getFamily());
            if (fontIndex == -1) {
                fontIndex = 0;
            }
        }
        famcomb.setSelectedIndex(fontIndex);
        final JPanel textFamilyPanel = new JPanel();
        final JLabel textFamilyLabel =
            new JLabel(JaxoLanguage.label("JaxoPrefsPanel.font.family"),
                JLabel.TRAILING);
        textFamilyPanel.add(textFamilyLabel);
        textFamilyPanel.add(famcomb);

        final JSpinner size = getSpinner(1, 1, 300, 2, null);
        this.sizecomb =
            new JaxoSpinnerComboBox(size,
                JaxoPSText.getFontSizeDefaults());
        sizecomb.setName("psfontsize");
        int fontSize;
        if (font == null) {
            fontSize = JaxoPrefs.getIntPref(JaxoPrefs.PREF_PSSIZE);
            size.setEnabled(false);
            sizecomb.setEnabled(false);
        } else {
            fontSize = font.getSize();
        }
        size.setValue(Integer.valueOf(fontSize));

        final JPanel textSizePanel = new JPanel();
        final JLabel textSizeLabel =
            new JLabel(JaxoLanguage.label("JaxoPrefsPanel.font.size"),
                JLabel.TRAILING);
        textSizePanel.add(textSizeLabel);
        textSizePanel.add(sizecomb);

        this.stycomb =
            new JComboBox(JaxoPSText.getFontStyles().toArray());
        stycomb.setName("psfontstyle");
        stycomb.setRenderer(new JaxoFontStyleRenderer());
        int fontStyle;
        if (font == null) {
            fontStyle = JaxoPrefs.getIntPref(JaxoPrefs.PREF_PSSTYLE);
            stycomb.setEnabled(false);
        } else {
            fontStyle = font.getStyle();
        }
        stycomb.setSelectedItem(Integer.valueOf(fontStyle));

        famcomb.addItemListener(new ItemListener() {
                public void itemStateChanged(final ItemEvent e) {
                    if (!size.isEnabled()) {
                        size.setEnabled(true);
                        sizecomb.setEnabled(true);
                        stycomb.setEnabled(true);
                    }
                }
            });

        final JPanel textStylePanel = new JPanel();
        final JLabel textStyleLabel =
            new JLabel(JaxoLanguage.label("JaxoPrefsPanel.font.style"),
                JLabel.TRAILING);
        textStylePanel.add(textStyleLabel);
        textStylePanel.add(stycomb);

        this.setLayout(new BorderLayout());
        this.setBorder(createI18NBorder("JaxoPSFontPanel.title"));

        this.add(BorderLayout.LINE_START, textFamilyPanel);
        this.add(BorderLayout.CENTER, textSizePanel);
        this.add(BorderLayout.LINE_END, textStylePanel);
    }

    /** Adds the given listener as a ChangeListener to the spinners of this panel.
     * If listener is null, does nothing.
     * @param listener The listener to add to this panel.
     */
    public void addEditPanelListener(final JaxoEditPanelListener listener) {
        if (listener != null) {
            famcomb.addItemListener(listener);
            sizecomb.addItemListener(listener);
            stycomb.addItemListener(listener);
        }
    }
}
