/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.io.exports;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import net.sf.jaxodraw.plugin.JaxoExportPlugin;
import net.sf.jaxodraw.util.JaxoInfo;


/** An abstract superclass for all built-in exporting classes.
 * @since 2.0
 */
public abstract class JaxoExport extends JaxoExportPlugin {
    private static List<Object> exportFormats;

    /** Protected empty constructor. */
    protected JaxoExport() {
        // protected to avoid public default constructor
        super();
    }

    /** {@inheritDoc}
     * @return As a built-in format, this always returns true.
     */
    public boolean makeAvailableAtRuntime() {
        return true;
    }

    /**
     * Returns a warning if the current graph contains a LaTeX text.
     * @return An (internationalized) warning.
     */
    public String getWarningForGraph() {
        if (getGraph().containsLatexText()) {
            return getLaTeXTextWarning();
        }

        return null;
    }

    /** Returns a list of export formats that are available within
     * the core of JaxoDraw by default. Elements are either
     * Lists (containing a subgroup of JaxoExport formats), or JaxoExports.
     * @return The list of built-in export formats.
     */
    public static List<Object> getBuiltInFormats() { // FIXME: make it <JaxoPlugin>
        synchronized (JaxoExport.class) {
            if (exportFormats == null) {
                exportFormats = new ArrayList<Object>(4);

                exportFormats.add(Arrays.asList(
                        new Object[]{
                            new JaxoExportLatexPS(),
                            new JaxoExportLatex()}));
                exportFormats.add(Arrays.asList(
                        new Object[]{
                            new JaxoExportPS(JaxoExportPS.Format.PS),
                            new JaxoExportPS(JaxoExportPS.Format.EPS)
                        }));
                exportFormats.add(Arrays.asList(
                        new Object[]{
                            new JaxoExportImg(JaxoExportImg.Format.JPG),
                            new JaxoExportImg(JaxoExportImg.Format.PNG)
                        }));

                exportFormats = Collections.unmodifiableList(exportFormats);
            }

            return exportFormats;
        }
    }

    /** {@inheritDoc} */
    public String version() {
        return JaxoInfo.VERSION_NUMBER;
    }
}
