/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.object.line;

import java.awt.geom.GeneralPath;

import net.sf.jaxodraw.object.JaxoObject;
import net.sf.jaxodraw.object.JaxoObjectEditPanel;


/** A fermion line.
 * @since 2.0
 */
public class JaxoFLine extends JaxoLineObject {
    private static final long serialVersionUID = 314159L;

    /** {@inheritDoc} */
    @Override
    public boolean isCopy(final JaxoObject comp) {
        boolean isCopy = false;

        if (comp instanceof JaxoFLine) {
            isCopy = super.isCopy(comp);
        }

        return isCopy;

    }

    /** {@inheritDoc} */
    public void prepareEditPanel(final JaxoObjectEditPanel editPanel) {
        editPanel.add2PointsPanel(getPoints(), 0, 0);
        editPanel.addArrowPanel(isPaintArrow(), isFlip(), getArrowPosition(), 1, 0);
        editPanel.addArrowOptionsEditPanel(getArrow(), 2, 0);
        editPanel.addDoubleLinePanel(isDoubleLine(), getDLSeparation(), 0, 1);
        editPanel.addStrokePanel(getStrokeWidth(), 1, 1);
        editPanel.addColorPanel(getColor(), JaxoObjectEditPanel.TYPE_LINE_COLOR, 2, 1);

        editPanel.setTitleAndIcon("Fermion_line_parameters", "fline.png");
    }

    /** {@inheritDoc} */
    protected GeneralPath getObjectPath() {
        final GeneralPath gp = getGeneralPath();
        gp.reset();
        if (isDoubleLine()) {
            final float length = (float) getRadius();
            final float sepx = getRelh() / length * getDLSeparation() / 2.f;
            final float sepy = getRelw() / length * getDLSeparation() / 2.f;
            gp.moveTo(getX() - sepx, getY() + sepy);
            gp.lineTo(getX2() - sepx, getY2() + sepy);
            gp.moveTo(getX() + sepx, getY() - sepy);
            gp.lineTo(getX2() + sepx, getY2() - sepy);
        } else {
            gp.moveTo(getX(), getY());
            gp.lineTo(getX2(), getY2());
        }
        return gp;
    }

    // Get the axodraw4j options set for this line

    /** {@inheritDoc} */
    protected String getAxo4JOptions(final float scale) {

        final StringBuffer optioncmd = new StringBuffer(128); //Assumes all option default to false in axodraw4j

        if (isPaintArrow()) {
            final float arpos = getArrowPosition();
            optioncmd.append(this.getArrow().latexCommand(arpos, scale));

            if (isFlip()) {
                optioncmd.append(",flip");
            }

            if (isDoubleLine()) {
                optioncmd.append(",double,sep=").append(
                   D_FORMAT.format(this.getDLSeparation()));
            }

            optioncmd.insert(0, '[').append(']');

        } else if (isDoubleLine()) {
            optioncmd.append("double,sep=").append(
                   D_FORMAT.format(this.getDLSeparation()));

            optioncmd.insert(0, '[').append(']');
        }

        return optioncmd.toString();
    }
}
