/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.object.loop;

import java.awt.geom.Ellipse2D;
import java.awt.geom.GeneralPath;

import net.sf.jaxodraw.object.JaxoObject;
import net.sf.jaxodraw.object.JaxoObjectEditPanel;


/** A fermion loop.
 * @since 2.0
 */
public class JaxoFLoop extends JaxoLoopObject {
    private static final long serialVersionUID = 314159L;

    /** {@inheritDoc} */
    @Override
    public final boolean isCopy(final JaxoObject comp) {
        boolean isCopy = false;

        if (comp instanceof JaxoFLoop) {
            isCopy = super.isCopy(comp);
        }

        return isCopy;

    }

    /** {@inheritDoc} */
    public void prepareEditPanel(final JaxoObjectEditPanel editPanel) {
        final int r = (int) Math.round(getRadius());

        editPanel.addXYRPanel(getX(), getY(), r, 0, 0);
        //editPanel.addArrowPanel(isPaintArrow(), isFlip(), false, false, 1, 0);
        editPanel.addArrowPanel(isPaintArrow(), isFlip(), getArrowPosition(), 1, 0);
        editPanel.addArrowOptionsEditPanel(getArrow(), 2, 0);
        editPanel.addStrokePanel(getStrokeWidth(), 0, 1);
        editPanel.addDoubleLinePanel(isDoubleLine(), getDLSeparation(), 1, 1);
        editPanel.addColorPanel(getColor(), JaxoObjectEditPanel.TYPE_LINE_COLOR, 2, 1);

        editPanel.setTitleAndIcon("Fermion_loop_parameters", "floop.png");
    }

    /** {@inheritDoc} */
    protected GeneralPath getObjectPath() {
        final GeneralPath gp = getGeneralPath();
        gp.reset();

        final double length = getRadius();
        final float dlsep = getDLSeparation() / 2.f;

        final Ellipse2D loop = new Ellipse2D.Double();
        if (isDoubleLine()) {
            loop.setFrame(-length - dlsep + getX(), -length - dlsep + getY(),
                2 * (length + dlsep), 2 * (length + dlsep));
            gp.append(loop, false);
            loop.setFrame(-length + dlsep + getX(), -length + dlsep + getY(),
                2 * (length - dlsep), 2 * (length - dlsep));
            gp.append(loop, false);
        } else {
            loop.setFrame(-length + getX(), -length + getY(), 2 * length,
                2 * length);
            gp.append(loop, false);
        }

        return gp;
    }

    // Get the axodraw4j options set for this loop

    /** {@inheritDoc} */
    protected String getAxo4JOptions(final float scale) {

        final StringBuffer optioncmd = new StringBuffer(128); //all loops are anti-clockwise

        if (isPaintArrow()) {
            final float arpos = getArrowPosition();
            optioncmd.append(this.getArrow().latexCommand(arpos, scale));

            if (isFlip()) {
                optioncmd.append(",flip");
            }

            if (isDoubleLine()) {
                optioncmd.append(",double,sep=").append(
                    D_FORMAT.format(this.getDLSeparation()));
            }

            optioncmd.insert(0, '[').append(']');

        } else if (isDoubleLine()) {
            optioncmd.append("[double,sep=").append(
                D_FORMAT.format(this.getDLSeparation())).append(']');
        }

        return optioncmd.toString();
    }
}
