/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.object.text;

import java.awt.geom.Point2D;

import net.sf.jaxodraw.object.JaxoHandle;
import net.sf.jaxodraw.object.JaxoObject;
import net.sf.jaxodraw.object.JaxoRotationObject;
import net.sf.jaxodraw.util.JaxoConstants;
import net.sf.jaxodraw.util.JaxoGeometry;
import net.sf.jaxodraw.util.graphics.JaxoGraphics2D;


/** Defines a JaxoObject which is just a string.
 * @since 2.0
 */
public abstract class JaxoTextObject extends JaxoObject
    implements JaxoRotationObject {

    private static final long serialVersionUID = 2L;

    /** The text string specifying this text object. */
    private String textString;

    /** The rotation angle of this text object. */
    private int rotationAngle;

    // Bean getter and setter methods

    /** Returns the textString property of this text object.
     * @return The textString property of this text object.
     */
    public final String getTextString() {
        return textString;
    }

    /** Sets the textString property of this text object.
     * @param string The textString property of this text object.
     */
    public void setTextString(final String string) {
        final String old = textString;
        this.textString = string;
        firePropertyChange("textString", old, textString);
    }

    /**
     * Returns the rotationAngle property of this text object.
     *
     * @return The rotationAngle property of this text object.
     */
    public final int getRotationAngle() {
        return rotationAngle;
    }

    /**
     * Sets the rotationAngle property of this text object.
     *
     * @param angle The rotationAngle property of this text object.
     */
    public final void setRotationAngle(final int angle) {
        final Integer old = Integer.valueOf(rotationAngle);
        this.rotationAngle = angle;
        firePropertyChange("rotationAngle", old, Integer.valueOf(rotationAngle));
    }

    /** {@inheritDoc} */
    @Override
    public boolean isCopy(final JaxoObject comp) {
        boolean isCopy = false;

        if (comp instanceof JaxoTextObject) {
            final JaxoTextObject tmp = (JaxoTextObject) comp;
            if ((tmp.getRotationAngle() == getRotationAngle())
                    && tmp.getTextString().equals(getTextString())
                    && super.isCopy(tmp)) {
                isCopy = true;
            }
        }

        return isCopy;
    }

    /** Sets all parameters from the given object to the current one.
     * @param temp The object to copy from.
     */
    public void copyFrom(final JaxoTextObject temp) {
        super.copyFrom(temp);
        setTextString(temp.getTextString());
        setRotationAngle(temp.getRotationAngle());
    }

    /** {@inheritDoc} */
    @Override
    public void setState(final JaxoObject o) {
        if (o instanceof JaxoTextObject) {
            copyFrom((JaxoTextObject) o);
        } else {
            throw new UnsupportedOperationException("Cannot copy from super type!");
        }
    }

    /** {@inheritDoc} */
    public final void paintHandles(final JaxoGraphics2D g2, final JaxoHandle h,
        final int editMode) {
        if (editMode == JaxoConstants.UNGROUP) {
            return;
        }

        h.paint(g2, getX(), getY(), isMarked(), !canBeSelected(SELECT_P1, editMode));
    }

    /** {@inheritDoc} */
    public boolean canBeSelected(final int handle, final int mode) {
        boolean active = (handle == SELECT_P1);
        if (mode == JaxoConstants.RESIZE) {
            active = false;
        }
        return active;
    }

    /** Returns the line width of this JaxoText.
     * This is not used for JaxoText.
     * @return Returns "%", corresponding to an empty comment line in LaTeX.
     */
    public final String latexWidth() {
        return "%";
    }

    /** {@inheritDoc} */
    public void rescaleObject(final int orx, final int ory, final float scale) {
        final Point2D newP =
            JaxoGeometry.scaledPoint(orx, ory, scale, getX(), getY());

        setLocation((int) Math.round(newP.getX()),
            (int) Math.round(newP.getY()));
    }

    /** {@inheritDoc} */
    @Override
    public void setPreferences() {
        super.setPreferences();
        setRotationAngle(0);
        setTextString(null);
    }
}
