/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.object.vertex;

import java.awt.Dimension;
import java.awt.Rectangle;
import java.awt.geom.Ellipse2D;
import java.awt.geom.Point2D;

import java.io.IOException;
import java.io.ObjectInputStream;

import net.sf.jaxodraw.object.JaxoObject;
import net.sf.jaxodraw.object.JaxoObjectEditPanel;
import net.sf.jaxodraw.util.graphics.JaxoGraphics2D;


/** A vertex (a filled circle).
 * @since 2.0
 */
public class JaxoVertexT1 extends JaxoVertex {
    private static final long serialVersionUID = 314159L;
    private transient Ellipse2D vertex = new Ellipse2D.Float();

    private void readObject(final ObjectInputStream in)
        throws IOException, ClassNotFoundException {
        in.defaultReadObject();
        vertex = new Ellipse2D.Float();
    }

    /** {@inheritDoc} */
    @Override
    public final boolean isCopy(final JaxoObject comp) {
        boolean isCopy = false;
        if (comp instanceof JaxoVertexT1) {
            isCopy = super.isCopy(comp);
        }

        return isCopy;
    }

    /** {@inheritDoc} */
    public final void paint(final JaxoGraphics2D g2) {
        updateVertex();

        g2.setColor(getColor());
        g2.setStroke(getStroke());

        g2.fill(vertex);
        g2.draw(vertex);
    }

    /**
     * Returns the bounding box of this object.
     *
     * @return the bounding box of this object.
     */
    public Rectangle getBounds() {
        updateVertex();

        return getStroke().createStrokedShape(vertex).getBounds();
    }

    /** {@inheritDoc} */
    public final String latexCommand(final float scale, final Dimension canvasDim) {
        final float radius = getLaTexRadius(scale);

        if ((int) radius == 0) {
            return "%";
        }

        final Point2D center = getLaTexCenter(scale, canvasDim.height);

        return "\\Vertex" + "(" + D_FORMAT.format(center.getX()) + ","
            + D_FORMAT.format(center.getY()) + ")" + "{"
            + D_FORMAT.format(radius) + "}";
    }

    /** {@inheritDoc} */
    public void prepareEditPanel(final JaxoObjectEditPanel editPanel) {
        prepareGenericPanel(editPanel, "vertexT1.png");
    }

    private void updateVertex() {
        final double length =
            Math.sqrt((getWidth() * getWidth()) + (getHeight() * getHeight()));
        vertex.setFrame(-length + getX(), -length + getY(), 2 * length,
            2 * length);
    }
}
