/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.object.vertex;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Rectangle;
import java.awt.geom.GeneralPath;

import net.sf.jaxodraw.object.Jaxo3PointObject;
import net.sf.jaxodraw.object.arrow.JaxoArrow.Coordinates;
import net.sf.jaxodraw.object.JaxoFillColorObject;
import net.sf.jaxodraw.object.JaxoObject;
import net.sf.jaxodraw.object.JaxoObjectEditPanel;
import net.sf.jaxodraw.util.JaxoColor;
import net.sf.jaxodraw.util.graphics.JaxoGraphics2D;


/** A vertex (a triangle).
 * @since 2.0
 */
public class JaxoVertexT5 extends Jaxo3PointObject implements JaxoFillColorObject {
    private static final long serialVersionUID = 314159L;
    /** The fill color of this JaxoVertexT5. */
    private Color fillColor;
    private boolean filled = true;

    /** {@inheritDoc} */
    public final boolean isFilled() {
        return filled;
    }

    /** {@inheritDoc} */
    public final void setFilled(final boolean value) {
        final Boolean old = Boolean.valueOf(filled);
        filled = value;
        firePropertyChange("filled", old, Boolean.valueOf(filled));
    }

    /** {@inheritDoc} */
    public final Color getFillColor() {
        return fillColor;
    }

    /** {@inheritDoc} */
    public final void setFillColor(final Color color) {
        final Color old = fillColor;
        this.fillColor = color;
        firePropertyChange("fillColor", old, fillColor);
    }

    /** Sets the points of this vertex.
     * @param sx1 The x coordinate of the first click point of this vertex.
     * @param sy1 The y coordinate of the first click point of this vertex.
     * @param sx2 The x coordinate of the second click point of this vertex.
     * @param sy2 The y coordinate of the second click point of this vertex.
     * @param sx3 The x coordinate of the third click point of this vertex.
     * @param sy3 The y coordinate of the third click point of this vertex.
     * @deprecated use
     * {@link net.sf.jaxodraw.object.Jaxo3PointObject#setLocation(int, int, int, int, int, int)}
     * instead.
     */
    public final void setVertexPts(final int sx1, final int sy1, final int sx2, final int sy2,
        final int sx3, final int sy3) {
        setLocation(sx1, sy1, sx2, sy2, sx3, sy3);
    }

    /** Sets all parameters from the given object to the current one.
     * @param temp The object to copy from.
     */
    public final void copyFrom(final JaxoVertexT5 temp) {
        super.copyFrom(temp);
        setFillColor(temp.getFillColor());
        setFilled(temp.isFilled());
    }

    /** {@inheritDoc} */
    @Override
    public void setState(final JaxoObject o) {
        if (o instanceof JaxoVertexT5) {
            copyFrom((JaxoVertexT5) o);
        } else {
            throw new UnsupportedOperationException("Cannot copy from super type!");
        }
    }

    /** {@inheritDoc} */
    @Override
    public final boolean isCopy(final JaxoObject comp) {
        boolean isCopy = false;

        if (comp instanceof JaxoVertexT5) {
            final JaxoVertexT5 vertex = (JaxoVertexT5) comp;
            if ((vertex.getFillColor().equals(getFillColor()))
                    && (vertex.isFilled() == isFilled())
                    && super.isCopy(vertex)) {
                isCopy = true;
            }
        }

        return isCopy;

    }

    /** {@inheritDoc} */
    public final void paint(final JaxoGraphics2D g2) {
        final GeneralPath gp = getObjectPath();

        if (isFilled()) {
            g2.setColor(getFillColor());
            g2.fill(gp);
        }

        g2.setColor(getColor());
        g2.setStroke(getStroke());
        g2.draw(gp);
    }

    /** {@inheritDoc} */
    @Override
    public void paintVisualAid(final JaxoGraphics2D g2) {
        // do nothing, needed to override ArcObject
    }

    /**
     * Returns the bounding box of this object.
     *
     * @return the bounding box of this object.
     */
    public Rectangle getBounds() {
        return getStroke().createStrokedShape(getObjectPath()).getBounds();
    }

    /** {@inheritDoc} */
    public final String latexCommand(final float scale, final Dimension canvasDim) {
        final double ax1 = getX() / scale;
        final double ay1 = (canvasDim.height - getY()) / scale;
        final double ax2 = getX2() / scale;
        final double ay2 = (canvasDim.height - getY2()) / scale;
        final double ax3 = getX3() / scale;
        final double ay3 = (canvasDim.height - getY3()) / scale;

        String command = "";

        if (isFilled()) {
            if (JaxoColor.isGrayScale(getFillColor())) {
                final float grayScale = JaxoColor.getGrayScaleFloat(getFillColor());

                command =
                    "\\GTri" + "(" + D_FORMAT.format(ax1) + ","
                    + D_FORMAT.format(ay1) + ")" + "(" + D_FORMAT.format(ax2)
                    + "," + D_FORMAT.format(ay2) + ")" + "("
                    + D_FORMAT.format(ax3) + "," + D_FORMAT.format(ay3) + ")"
                    + "{" + GRAY_SCALE_FORMAT.format(grayScale) + "}";
            } else {
                final String tlc = JaxoColor.getColorName(getColor());
                final String tfc = JaxoColor.getColorName(getFillColor());

                command =
                    "\\CTri" + "(" + D_FORMAT.format(ax1) + ","
                    + D_FORMAT.format(ay1) + ")" + "(" + D_FORMAT.format(ax2)
                    + "," + D_FORMAT.format(ay2) + ")" + "("
                    + D_FORMAT.format(ax3) + "," + D_FORMAT.format(ay3) + ")"
                    + "{" + tlc + "}" + "{" + tfc + "}";
            }
        } else {
            command =
                "\\ETri" + "(" + D_FORMAT.format(ax1) + ","
                + D_FORMAT.format(ay1) + ")" + "(" + D_FORMAT.format(ax2)
                + "," + D_FORMAT.format(ay2) + ")" + "("
                + D_FORMAT.format(ax3) + "," + D_FORMAT.format(ay3) + ")";
        }

        return command;
    }

    /** {@inheritDoc} */
    @Override
    public void setPreferences() {
        super.setPreferences();
        setFilled(true);
        setFillColor(JaxoColor.getDefaultFillColor(getColor()));
    }


    private GeneralPath getObjectPath() {
        final GeneralPath gp = getGeneralPath();
        gp.reset();

        gp.moveTo(getX(), getY());
        gp.lineTo(getX2(), getY2());
        gp.lineTo(getX3(), getY3());
        gp.closePath();

        return gp;
    }

   /** {@inheritDoc} */
    public void prepareEditPanel(final JaxoObjectEditPanel editPanel) {
        prepareT5Panel(editPanel);
    }

    private void prepareT5Panel(final JaxoObjectEditPanel editPanel) {
        editPanel.add3PointsPanel(getPoints(), 0, 0);
        editPanel.addStrokePanel(getStrokeWidth(), 0, 1);
        editPanel.addFillLineColorPanels(getColor(), getFillColor(),
            isFilled(), 1, 1, 1, 0);

        editPanel.setTitleAndIcon("Vertex_parameters", "vertexT5.png");
    }

   /**
    * {@inheritDoc}
    *
    * @return null a vertex doesn't have an arrow.
    */
    public Coordinates arrowCoordinates() {
        return null;
    }
}
