/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */

package net.sf.jaxodraw.plugin;

/**
 * A wrapper of JaxoDictionary for plugins.
 */
public interface JaxoPluginDictionary {

    /**
     * Translate 'key' as a label (typically, by appending ": " to the
     * default translation).
     *
     * @param key The key to translate.
     *
     * @return The translation.
     */
    String label(String key);

    /**
     * Translates the given string into the current default language.
     *
     * @param key The key to translate.
     *
     * @return The translated string, or an empty string if key itself is empty.
     */
    String translate(String key);

    /**
     * Translates the given string into the current default language,
     * interpolating a single argument.
     *
     * Equivalent to {@link #message(String,Object[])} with second argument
     * <code>new Object[] {argument}</code>.
     *
     * @param key The key to translate.
     * @param argument An object.
     *
     * @return The translated string.
     */
    String message(String key, Object argument);

    /**
     * Translates the given string into the current default language,
     * interpolating two arguments.
     *
     * Equivalent to {@link #message(String,Object[])} with second argument
     * <code>new Object[] {argument1, argument2}</code>.
     *
     * @param key The key to translate.
     * @param argument1 A first object.
     * @param argument2 A second object.
     *
     * @return The translated string.
     */
    String message(String key, Object argument1, Object argument2);

    /**
     * Translates the given string into the current default language,
     * interpolating arguments.
     *
     * <p>
     * If <code>arguments</code> is not empty, the translation of
     * <code>key</code> is used as a
     * {@link java.text.MessageFormat} pattern, the <code>arguments</code>
     * array is then used as the argument for
     * {@link java.text.MessageFormat#format(Object)}.
     * </p>
     *
     * <p>
     * See the language.properties for notes on the parsing of
     * MessageFormat patterns.
     * </p>
     *
     * <p>
     * If however, <code>arguments</code> is empty, the key
     * translation is not parsed as MessageFormat pattern (that way, the
     * <code>message</code> methods can be used generically.)
     * </p>
     *
     * <p>
     * <em>By convention</em>, the keys for MessageFormat pattern (and only
     * them) contain percentage signs, followed by a number (starting from zero,
     * as MessageFormat), to denote the occurrence of arguments (e.g.
     * <code>JaxoIO.read%0ReadFailed=File "{0}" could not be read.</code>).
     * Then:
     * </p>
     *
     * <ul>
     * <li>In the properties file, it is clear from the key whether the value
     * is a MessageFormat pattern or a plain String.</li>
     * <li>In the java source, one can see whether the actual number of arguments
     * matches the expected one (e.g.
     * <code>message("JaxoIO.read%0ReadFailed", new Object[] { a, b});</code>
     * can be seen to be incorrect.</li>
     * </ul>
     *
     * @param key The key to translate.
     * @param arguments An array of objects (arguments for MessageFormat).
     *
     * @return The translated string.
     */
    String message(String key, Object[] arguments);
}
