/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.util;

import java.util.Locale;

/** Translation using JaxoLanguage within a given context (bundleID and/or
    prefix for the keys). All methods
    call eventually into JaxoLanguage.
 */
public class JaxoDictionary {
    private String bundleID;
    private String prefix;

    /** Create for the given class. JaxoLanguage will be used to
        lookup the bundle containing the translations.
     * @param c The class to register.
    */
    public JaxoDictionary(final Class<?> c) {
        JaxoLanguage.registerBundle(c, this);
    }

    /* package */ void setValues(final String id, final String pfix) {
        this.bundleID = id;
        this.prefix = pfix;
    }

    // Key starting with '/' is a global "well-known" key
    // Other non-empty keys get the prefix ...
    private String key(final String key) {
        return (key.length() == 0 || key.charAt(0) == '/') ? key : (prefix + key);
    }

    /** Locale corresponding to the current language.
     * @return The current locale.
     */
    public Locale locale() {
        return JaxoLanguage.locale();
    }

    /** Translate 'key' as a label (typically, by appending ": " to the
        default translation).
     * @param key The key to translate.
     * @return The translation.
     */
    public String label(final String key) {
        return JaxoLanguage.message("JaxoDraw.label%0", value(key));
    }

    /** Translates the given string into the current default language.
     * @param key The key to translate.
     * @return The translated string, or an empty string if key itself is empty.
     */
    public String value(final String key) {
        return JaxoLanguage.bundleTranslate(key(key), bundleID);
    }

    /** Translates the given string into the current default language,
        interpolating a single argument.
        Equivalent to {@link #message(String,Object[])} with second argument
        <code>new Object[] { argument }</code>.
     * @param key The key to translate.
     * @param argument An object.
     * @return The translated string.
     */
    public String message(final String key, final Object argument) {
        return JaxoLanguage.bundleMessage(key(key), bundleID, new Object[]{argument});
    }

    /** Translates the given string into the current default language,
        interpolating two arguments.
        Equivalent to {@link #message(String,Object[])} with second argument
        <code>new Object[] { argument1, argument2 }</code>.
     * @param key The key to translate.
     * @param argument1 A first object.
     * @param argument2 A second object.
     * @return The translated string.
     */
    public String message(final String key, final Object argument1, final Object argument2) {
        return JaxoLanguage.bundleMessage(key(key), bundleID, new Object[]{argument1, argument2});
    }

    /** Translates the given string into the current default language,
        interpolating arguments.
        <p>If <code>arguments</code> is not empty, the translation of
        <code>key</code> is used as a
        {@link java.text.MessageFormat} pattern, the <code>arguments</code>
        array is then used as the argument for
        {@link java.text.MessageFormat#format(Object)}.</p>
        <p>See the language.properties for notes on the parsing of
        MessageFormat patterns.
        <p>If however, <code>arguments</code> is empty, the key
        translation is not parsed as MessageFormat pattern (That way, the
        <code>message</code> methods can be used generically.)
        <p><em>By convention</em>, the keys for MessageFormat pattern (and only
        them) contain percentage signs, followed by a number (starting from zero,
        as MessageFormat), to denote the occurrence of arguments (e.g.
        <code>JaxoIO.read%0ReadFailed=File "{0}" could not be read.</code>).
        Then:</p>
        <ul><li>In the properties file, it is clear from the key whether the value
        is a MessageFormat pattern or a plain String.
        <li>In the java source, one can see whether the actual number of arguments
        matches the expected one (e.g.
        <code>message("JaxoIO.read%0ReadFailed", new Object[] { a, b});</code>
        can be seen to be incorrect.
        </ul>
     * @param key The key to translate.
     * @param arguments An array of objects (arguments for MessageFormat).
     * @return The translated string.
     */
    public String message(final String key, final Object[] arguments) {
        return JaxoLanguage.bundleMessage(key(key), bundleID, arguments);
    }
}
