/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.util;

import java.awt.Component;
import java.awt.Rectangle;
import java.awt.Window;

/**
 * Policy for placing windows.
 */
public interface Location {

    /**
     * Set the location of the given window.
     *
     * @param w the window to locate.
     */
    void setLocation(Window w);

    /**
     * Class to set locations relative to other components.
     */
    class RelativeTo implements Location {

        private final Component component;

        /**
         * Constructor.
         *
         * @param c the component that serves as a reference point.
         */
        public RelativeTo(Component c) {
            super();
            component = c;
        }

        /** {@inheritDoc} */
        public void setLocation(Window w) {
            w.setLocationRelativeTo(component);
        }
    }

    /**
     * Class to set locations relative to other components.
     */
    class RelativeToAvoiding implements Location {

        private final Component component;
        private final Rectangle avoid;

        /**
         * Constructor.
         *
         * @param c the component that serves as a reference.
         * @param avoidr the component to avoid.
         */
        public RelativeToAvoiding(Component c, Rectangle avoidr) {
            super();
            this.component = c;
            this.avoid = new Rectangle(avoidr);
        }

        /** {@inheritDoc} */
        public void setLocation(Window w) {
            JaxoGeometry.setLocationRelativeToAvoiding(w, component, avoid);
        }
    }
}
