/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.gui.grid;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Point;

import net.sf.jaxodraw.AbstractJaxoBaseTest;
import net.sf.jaxodraw.util.JaxoColor;
import net.sf.jaxodraw.util.JaxoPrefs;

/**
 * Tests for JaxoDefaultGrid.
 */
public class JaxoGridTest extends AbstractJaxoBaseTest {
    private final Dimension gridSize = new Dimension(800, 600);

    /** Constructor: just calls super().
     * @param testName The name of this test.
     */
    public JaxoGridTest(String testName) {
        super(testName);
    }

    /**
     * Test of snapX method, of class JaxoDraw.JaxoDefaultGrid.
     */
    public void testSnap() {
        JaxoDefaultGrid grid = new JaxoDefaultGrid(10, JaxoPaintableGrid.TYPE_RECTANGULAR, gridSize);
        Point testPoint = new Point(7, 12);
        assertEquals(new Point(10, 10), grid.snappedPoint(testPoint));

        assertFalse(grid.isSnapped(testPoint));
        testPoint.setLocation(10, 10);
        assertTrue(grid.isSnapped(testPoint));

        testPoint.setLocation(7, 12);

        grid.setGridSize(20);
        assertEquals(new Point(0, 20), grid.snappedPoint(testPoint));
        grid.setGridType(JaxoPaintableGrid.TYPE_HEXAGONAL);
        testPoint.setLocation(17, 27);
        assertEquals(new Point(20, 34), grid.snappedPoint(testPoint));
    }

    /**
     * Tests constructor of class JaxoDraw.JaxoDefaultGrid.
     */
    public void testConstructor() {
        JaxoPaintableGrid grid = new JaxoDefaultGrid(15, JaxoPaintableGrid.TYPE_HEXAGONAL, gridSize);
        assertEquals(16, grid.getGridSize());
        assertEquals(JaxoPaintableGrid.TYPE_HEXAGONAL, grid.getGridType());
    }

    /**
     * Tests copyFrom of class JaxoDraw.JaxoDefaultGrid.
     */
    public void testCopyFrom() {
        JaxoDefaultGrid grid = new JaxoDefaultGrid(10, JaxoPaintableGrid.TYPE_HEXAGONAL,
                JaxoPaintableGrid.STYLE_CROSS, Color.YELLOW, gridSize);
        grid.setBackground(Color.CYAN);
        JaxoDefaultGrid copy = new JaxoDefaultGrid(20, JaxoPaintableGrid.TYPE_RECTANGULAR,
                JaxoPaintableGrid.STYLE_LINE, Color.PINK, new Dimension());
        copy.setBackground(Color.MAGENTA);

        copy.copyFrom(grid);

        assertEquals(10, copy.getGridSize());
        assertEquals(JaxoPaintableGrid.TYPE_HEXAGONAL, copy.getGridType());
        assertEquals(JaxoPaintableGrid.STYLE_CROSS, copy.getGridStyle());
        assertEquals(Color.YELLOW, copy.getGridColor());
        assertEquals(gridSize, copy.getCanvasSize());
        assertEquals(Color.CYAN, copy.getBackground());
    }

    /**
     * Tests newDefaultGrid of class JaxoDraw.JaxoDefaultGrid.
     */
    public void testNewDefaultGrid() {
        JaxoDefaultGrid grid = JaxoDefaultGrid.newDefaultGrid();
        assertEquals(JaxoPrefs.getIntPref(JaxoPrefs.PREF_GRIDSIZE), grid.getGridSize());
        assertEquals(JaxoPrefs.getStringPref(JaxoPrefs.PREF_GRIDTYPE),
                grid.getGridType() == JaxoPaintableGrid.TYPE_RECTANGULAR
                ? "rectangular"
                : "hexagonal");
        assertEquals(JaxoPrefs.getIntPref(JaxoPrefs.PREF_GRIDSTYLE), grid.getGridStyle());
        assertEquals(JaxoColor.getColor(JaxoPrefs.getStringPref(JaxoPrefs.PREF_GRIDCOLOR),
                JaxoColor.ALL_COLORS_MODE), grid.getGridColor());
    }
}
