/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.gui.panel;

import java.awt.Point;

import java.beans.PropertyChangeEvent;

import java.io.File;
import junit.framework.TestCase;

import net.sf.jaxodraw.gui.JaxoTab;
import net.sf.jaxodraw.gui.JaxoTestCanvas;
import net.sf.jaxodraw.io.JaxoIO;
import net.sf.jaxodraw.util.JaxoDictionary;

/** Tests for JaxoStatusBar. */
public class JaxoStatusBarTest extends TestCase {
    private static final JaxoDictionary LANGUAGE = new JaxoDictionary(JaxoStatusBar.class);
    /** Constructor: just calls super().
     * @param testName The name of this test.
     */
    public JaxoStatusBarTest(String testName) {
        super(testName);
    }

    /** Tests JaxoStatusBar with a null tab. */
    public void testDefaults() {
        JaxoStatusBar statusBar = new JaxoStatusBar();
        String expected = LANGUAGE.value("mouseCoordinates_none");
        String actual = statusBar.getCurrentCoordinateText();
        assertEquals("Coordinate text not correct!", expected, actual);

        expected = LANGUAGE.value("mode_none");
        actual = statusBar.getCurrentModeText();
        assertEquals("Mode text not correct!", expected, actual);

        int max = 46;
        String filename = JaxoIO.currentDirectoryString();
        if (filename.length() > max) {
            expected = LANGUAGE.message(
                    "directory%0ending",
                    filename.substring(filename.length() - max));
        } else {
            expected = LANGUAGE.message("directory%0", filename);
        }
        actual = statusBar.getCurrentFileText();
        assertEquals("Coordinate text not correct!", expected, actual);
    }

    /** Test of propertyChange method in JaxoDraw.JaxoStatusBar. */
    public void testPropertyChange() {
        Object source = (Object) this;
        String name = "Jaxo.mouseLocation";
        Object oldOb = null;
        Point point = new Point(20, 30);
        Object newOb = (Object) point;
        PropertyChangeEvent event =
            new PropertyChangeEvent(source, name, oldOb, newOb);

        JaxoStatusBar statusBar = new JaxoStatusBar();
        statusBar.propertyChange(event);
        String expected = LANGUAGE.message(
            "mouseCoordinates%0%1", new Integer(point.x),
                new Integer(point.y));
        String actual = statusBar.getCurrentCoordinateText();
        assertEquals("Coordinate text not correct!", expected, actual);

        JaxoTab tab = new JaxoTab(new JaxoTestCanvas());
        File saveFile =
                new File(JaxoIO.currentDirectoryString(), "saveFileName");
        String saveFileName = saveFile.getAbsolutePath();
        tab.setSaveFileName(saveFileName);
        newOb = (Object) tab;
        name = "Jaxo.selectedTab";
        event = new PropertyChangeEvent(source, name, oldOb, newOb);
        statusBar.propertyChange(event);
        String filename = JaxoIO.directoryString(saveFileName);
        expected = LANGUAGE.message("directory%0", filename);

        int max = 46;
        if (filename.length() > max) {
            expected = LANGUAGE.message("directory%0ending",
                    filename.substring(filename.length() - max));
        }

        actual = statusBar.getCurrentFileText();
        assertEquals("File text not correct!", expected, actual);
    }
}
