/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.gui.panel.button;

import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import junit.framework.TestCase;

import net.sf.jaxodraw.JaxoTestUtils;
import net.sf.jaxodraw.util.JaxoConstants;
import net.sf.jaxodraw.util.JaxoLanguage;

/** Tests for JaxoEditButtons. */
public class JaxoEditButtonsTest extends TestCase {

    private static final int[] EDIT_MODES =
    {
        JaxoConstants.MOVE, JaxoConstants.RESIZE, JaxoConstants.COLOR,
        JaxoConstants.EDIT, JaxoConstants.DUPLICATE, JaxoConstants.BACKGROUND,
        JaxoConstants.FOREGROUND, JaxoConstants.DELETE, JaxoConstants.SELECT,
        JaxoConstants.UNGROUP, JaxoConstants.DUMMIE, JaxoConstants.DUMMIE1
    };
    private static final int NOF_BUTTONS = EDIT_MODES.length;
    private static final String[] TOOLTIP_KEYS =
    {
        "Move_an_object", "Resize_an_object", "Color_an_object",
        "Edit_an_object", "Duplicate_an_object",
        "Put_an_object_in_the_background", "Put_an_object_in_the_foreground",
        "Delete_an_object", "Select_objects", "Ungroup_objects", "", ""
    };

    private final JaxoEditButtons editButtons = new JaxoEditButtons();
    private String actionCommand;

    /** Constructor: just calls super().
     * @param testName The name of this test.
     */
    public JaxoEditButtonsTest(String testName) {
        super(testName);
    }

    /** Tests constructor. */
    public void testConstructor() {
        for (int i = 0; i < NOF_BUTTONS; i++) {
            String name = JaxoConstants.getModeAsString(EDIT_MODES[i]);
            Component component = JaxoTestUtils.getChildNamed(editButtons, name);
            assertNotNull(component);
            JaxoPanelButton button = (JaxoPanelButton) component;
            assertEquals(JaxoConstants.getModeAsString(EDIT_MODES[i]),
                button.getName());
            assertEquals(JaxoLanguage.translate(TOOLTIP_KEYS[i]),
                button.getToolTipText());
            if (i >= NOF_BUTTONS - 2) {
                // dummy button is disabled
                assertFalse(button.isEnabled());
            }
        }
    }

    /** Tests pressButton. */
    public void testPressButton() {
        for (int i = 0; i < NOF_BUTTONS; i++) {
            editButtons.pressButton(EDIT_MODES[i]);
            for (int j = 0; j < NOF_BUTTONS; j++) {
                String name = JaxoConstants.getModeAsString(EDIT_MODES[j]);
                JaxoPanelButton button =
                    (JaxoPanelButton) JaxoTestUtils.getChildNamed(editButtons, name);
                if (i == j && i < NOF_BUTTONS - 2) {
                    assertTrue("should be selected: " + name,
                        button.isSelected());
                } else {
                    assertFalse("should NOT be selected: " + name,
                        button.isSelected());
                }
            }
        }
    }

    /** Tests setDefault. */
    public void testSetDefault() {
        for (int i = 0; i < NOF_BUTTONS; i++) {
            editButtons.setDefault(EDIT_MODES[i]);
            for (int j = 0; j < NOF_BUTTONS; j++) {
                String name = JaxoConstants.getModeAsString(EDIT_MODES[j]);
                JaxoPanelButton button =
                    (JaxoPanelButton) JaxoTestUtils.getChildNamed(editButtons, name);
                if (i == j && i < NOF_BUTTONS - 2) {
                    assertTrue("should be default: " + name,
                        button.isDefault());
                } else {
                    assertFalse("should NOT be default: " + name,
                        button.isDefault());
                }
            }
        }
    }

    /** Tests updateLanguage. */
    public void testUpdateLanguage() {
        editButtons.updateLanguage();
        for (int i = 0; i < NOF_BUTTONS - 1; i++) {
            String name = JaxoConstants.getModeAsString(EDIT_MODES[i]);
            JaxoPanelButton button =
                (JaxoPanelButton) JaxoTestUtils.getChildNamed(editButtons, name);
            assertEquals(JaxoLanguage.translate(TOOLTIP_KEYS[i]),
                button.getToolTipText());
        }
    }

    /** Tests addActionListener. */
    public void testAddActionListener() {
        editButtons.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent evt) {
                actionCommand = evt.getActionCommand();
            }
        });

        for (int i = 0; i < NOF_BUTTONS - 2; i++) {
            String name = JaxoConstants.getModeAsString(EDIT_MODES[i]);
            JaxoPanelButton button =
                (JaxoPanelButton) JaxoTestUtils.getChildNamed(editButtons, name);
            actionCommand = null;
            button.doClick();
            assertEquals("wrong action command for button #" + i, name, actionCommand);
        }
    }

}
