/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.gui.panel.button;

import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import java.beans.PropertyChangeEvent;

import junit.framework.TestCase;

import net.sf.jaxodraw.JaxoTestUtils;
import net.sf.jaxodraw.util.JaxoConstants;
import net.sf.jaxodraw.util.JaxoLanguage;

/** Tests for JaxoGridButtons. */
public class JaxoGridButtonsTest extends TestCase {

    private static final int[] GRID_MODES =
    {
        JaxoConstants.GRID, JaxoConstants.ZOOM, JaxoConstants.DUMMIE,
        JaxoConstants.QUIT
    };
    private static final int NOF_BUTTONS = GRID_MODES.length;
    private static final String[] TOOLTIP_KEYS =
    {"Snap_points_to_the_grid", "Dynamical_zoom", "", "Exits_JaxoDraw"};

    private final JaxoGridButtons gridButtons = new JaxoGridButtons();
    private String actionCommand;

    /** Constructor: just calls super().
     * @param testName The name of this test.
     */
    public JaxoGridButtonsTest(String testName) {
        super(testName);
    }

    /** Tests constructor. */
    public void testConstructor() {
        for (int i = 0; i < NOF_BUTTONS; i++) {
            String name = JaxoConstants.getModeAsString(GRID_MODES[i]);
            Component component = JaxoTestUtils.getChildNamed(gridButtons, name);
            assertNotNull(component);
            JaxoPanelButton button = (JaxoPanelButton) component;
            assertEquals(JaxoConstants.getModeAsString(GRID_MODES[i]),
                button.getName());
            assertEquals(JaxoLanguage.translate(TOOLTIP_KEYS[i]),
                button.getToolTipText());
            if (i == 2) {
                // dummy button is disabled
                assertFalse(button.isEnabled());
            }
        }
    }

    /** Tests pressButton. */
    public void testPressButton() {
        // first enable all buttons
        PropertyChangeEvent event =
            new PropertyChangeEvent(this, "Jaxo.gridOn", Boolean.FALSE, Boolean.TRUE);
        gridButtons.propertyChange(event);
        event =
            new PropertyChangeEvent(this, "Jaxo.snap", Boolean.FALSE, Boolean.TRUE);
        gridButtons.propertyChange(event);

        for (int i = 0; i < NOF_BUTTONS; i++) {
            if (i == 2) {
                continue;
            }
            gridButtons.pressButton(GRID_MODES[i]);
            for (int j = 0; j < NOF_BUTTONS; j++) {
                String name = JaxoConstants.getModeAsString(GRID_MODES[j]);
                JaxoPanelButton button =
                    (JaxoPanelButton) JaxoTestUtils.getChildNamed(gridButtons, name);
                if (i == j && JaxoConstants.isGridMode(GRID_MODES[i]) && j != 3) {
                    assertTrue("should be selected: " + name,
                        button.isSelected());
                } else {
                    assertFalse("should NOT be selected: " + name,
                        button.isSelected());
                }
            }
            if (i == 0) {
                // switch off the snap again
                event = new PropertyChangeEvent(
                    this, "Jaxo.snap", Boolean.TRUE, Boolean.FALSE);
                gridButtons.propertyChange(event);
            }
        }
        gridButtons.pressButton(JaxoConstants.MOVE);
        for (int i = 0; i < NOF_BUTTONS; i++) {
            if (i == 2) {
                continue;
            }
            String name = JaxoConstants.getModeAsString(GRID_MODES[i]);
                JaxoPanelButton button =
                    (JaxoPanelButton) JaxoTestUtils.getChildNamed(gridButtons, name);
            assertFalse("should NOT be selected: " + name, button.isSelected());
        }
    }

    /** Tests setDefault. */
    public void testSetDefault() {
        for (int i = 0; i < NOF_BUTTONS; i++) {
            if (i == 2) {
                continue;
            }
            gridButtons.setDefault(GRID_MODES[i]);
            for (int j = 0; j < NOF_BUTTONS; j++) {
                String name = JaxoConstants.getModeAsString(GRID_MODES[j]);
                JaxoPanelButton button =
                    (JaxoPanelButton) JaxoTestUtils.getChildNamed(gridButtons, name);
                if (i == j && JaxoConstants.isGridMode(GRID_MODES[i])) {
                    assertTrue("should be default: " + name,
                        button.isDefault());
                } else {
                    assertFalse("should NOT be default: " + name,
                        button.isDefault());
                }
            }
        }
        gridButtons.setDefault(JaxoConstants.MOVE);
        for (int i = 0; i < NOF_BUTTONS; i++) {
            if (i == 2) {
                continue;
            }
            String name = JaxoConstants.getModeAsString(GRID_MODES[i]);
                JaxoPanelButton button =
                    (JaxoPanelButton) JaxoTestUtils.getChildNamed(gridButtons, name);
            assertFalse("should NOT be default: " + name, button.isDefault());
        }
    }

    /** Tests updateLanguage. */
    public void testUpdateLanguage() {
        gridButtons.updateLanguage();
        for (int i = 0; i < NOF_BUTTONS; i++) {
            String name = JaxoConstants.getModeAsString(GRID_MODES[i]);
            JaxoPanelButton button =
                (JaxoPanelButton) JaxoTestUtils.getChildNamed(gridButtons, name);
            assertEquals(JaxoLanguage.translate(TOOLTIP_KEYS[i]),
                button.getToolTipText());
        }
    }

    /** Tests addActionListener. */
    public void testAddActionListener() {
        gridButtons.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent evt) {
                actionCommand = evt.getActionCommand();
            }
        });

        // first enable all buttons
        PropertyChangeEvent event = new PropertyChangeEvent(this, "Jaxo.gridOn",
            Boolean.FALSE, Boolean.TRUE);
        gridButtons.propertyChange(event);
        event = new PropertyChangeEvent(this, "Jaxo.snap",
            Boolean.FALSE, Boolean.TRUE);
        gridButtons.propertyChange(event);
        // this should do nothing
        event = new PropertyChangeEvent(this, "Jaxo.canDoAnything",
            Boolean.FALSE, Boolean.TRUE);
        gridButtons.propertyChange(event);

        for (int i = 0; i < NOF_BUTTONS; i++) {
            if (i == 2) {
                continue;
            }
            String name = JaxoConstants.getModeAsString(GRID_MODES[i]);
            JaxoPanelButton button =
                (JaxoPanelButton) JaxoTestUtils.getChildNamed(gridButtons, name);
            actionCommand = null;
            button.doClick();
            assertEquals("wrong action command!", name, actionCommand);
        }
    }

}
