/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.gui.panel.button;

import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import java.beans.PropertyChangeEvent;

import junit.framework.TestCase;

import net.sf.jaxodraw.JaxoTestUtils;
import net.sf.jaxodraw.util.JaxoConstants;
import net.sf.jaxodraw.util.JaxoLanguage;

/** Tests for JaxoMiscButtons. */
public class JaxoMiscButtonsTest extends TestCase {

    private static final int[] MISC_MODES =
    {
        JaxoConstants.ZIGZAG, JaxoConstants.BOX, JaxoConstants.BLOB,
        JaxoConstants.VERTEX, JaxoConstants.TEXT, JaxoConstants.LATEX,
        JaxoConstants.DUMMIE, JaxoConstants.DUMMIE
    };
    private static final int NOF_BUTTONS = MISC_MODES.length;
    private static final String[] TOOLTIP_KEYS =
    {
        "Draw_a_zigzag_line", "Draw_a_box", "Draw_a_blob", "Draw_a_vertex",
        "Draw_a_text_string", "Draw_a_LaTex_text", "", ""
    };

    private final JaxoMiscButtons miscButtons = new JaxoMiscButtons();
    private String actionCommand;

    /** Constructor: just calls super().
     * @param testName The name of this test.
     */
    public JaxoMiscButtonsTest(String testName) {
        super(testName);
    }

    /** Tests constructor. */
    public void testConstructor() {
        for (int i = 0; i < NOF_BUTTONS; i++) {
            String name = JaxoConstants.getModeAsString(MISC_MODES[i]);
            Component component = JaxoTestUtils.getChildNamed(miscButtons, name);
            assertNotNull(component);
            JaxoPanelButton button = (JaxoPanelButton) component;
            assertEquals(JaxoConstants.getModeAsString(MISC_MODES[i]),
                button.getName());
            assertEquals(JaxoLanguage.translate(TOOLTIP_KEYS[i]),
                button.getToolTipText());
            if (i == NOF_BUTTONS - 1 || i == NOF_BUTTONS - 2) {
                // dummy button is disabled
                assertFalse(button.isEnabled());
            }
        }
    }

    /** Tests pressButton. */
    public void testPressButton() {
        for (int i = 0; i < NOF_BUTTONS; i++) {
            miscButtons.pressButton(MISC_MODES[i]);
            for (int j = 0; j < NOF_BUTTONS; j++) {
                String name = JaxoConstants.getModeAsString(MISC_MODES[j]);
                JaxoPanelButton button =
                    (JaxoPanelButton) JaxoTestUtils.getChildNamed(miscButtons, name);
                if (i == j && i != NOF_BUTTONS - 1 && i != NOF_BUTTONS - 2) {
                    assertTrue("should be selected: " + name,
                        button.isSelected());
                } else {
                    assertFalse("should NOT be selected: " + name,
                        button.isSelected());
                }
            }
        }
    }

    /** Tests setDefault. */
    public void testSetDefault() {
        for (int i = 0; i < NOF_BUTTONS; i++) {
            miscButtons.setDefault(MISC_MODES[i]);
            for (int j = 0; j < NOF_BUTTONS; j++) {
                String name = JaxoConstants.getModeAsString(MISC_MODES[j]);
                JaxoPanelButton button =
                    (JaxoPanelButton) JaxoTestUtils.getChildNamed(miscButtons, name);
                if (i == j && i != NOF_BUTTONS - 1 && i != NOF_BUTTONS - 2) {
                    assertTrue("should be default: " + name,
                        button.isDefault());
                } else {
                    assertFalse("should NOT be default: " + name,
                        button.isDefault());
                }
            }
        }
    }

    /** Tests updateLanguage. */
    public void testUpdateLanguage() {
        miscButtons.updateLanguage();
        for (int i = 0; i < NOF_BUTTONS - 2; i++) {
            String name = JaxoConstants.getModeAsString(MISC_MODES[i]);
            JaxoPanelButton button =
                (JaxoPanelButton) JaxoTestUtils.getChildNamed(miscButtons, name);
            assertEquals(JaxoLanguage.translate(TOOLTIP_KEYS[i]),
                button.getToolTipText());
        }
    }

    /** Tests addActionListener. */
    public void testAddActionListener() {
        miscButtons.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent evt) {
                actionCommand = evt.getActionCommand();
            }
        });

        for (int i = 0; i < NOF_BUTTONS - 2; i++) {
            String name = JaxoConstants.getModeAsString(MISC_MODES[i]);
            JaxoPanelButton button =
                (JaxoPanelButton) JaxoTestUtils.getChildNamed(miscButtons, name);
            actionCommand = null;
            button.doClick();
            assertEquals("wrong action command!", name, actionCommand);
        }
    }

    /** Tests propertyChange. */
    public void testPropertyChange() {
        PropertyChangeEvent event =
            new PropertyChangeEvent(this, "Jaxo.vertexType",
                new Integer(JaxoConstants.VERTEXT1),
                new Integer(JaxoConstants.VERTEXT2));
        miscButtons.propertyChange(event);
        event =
            new PropertyChangeEvent(this, "Jaxo.vertexType",
                new Integer(JaxoConstants.VERTEXT2),
                new Integer(JaxoConstants.VERTEXT3));
        miscButtons.propertyChange(event);
        event =
            new PropertyChangeEvent(this, "Jaxo.vertexType",
                new Integer(JaxoConstants.VERTEXT3),
                new Integer(JaxoConstants.VERTEXT4));
        miscButtons.propertyChange(event);
        event =
            new PropertyChangeEvent(this, "Jaxo.vertexType",
                new Integer(JaxoConstants.VERTEXT4),
                new Integer(JaxoConstants.VERTEXT5));
        miscButtons.propertyChange(event);
        event =
            new PropertyChangeEvent(this, "Jaxo.vertexType",
                new Integer(JaxoConstants.VERTEXT5),
                new Integer(JaxoConstants.VERTEXT6));
        miscButtons.propertyChange(event);
        event =
            new PropertyChangeEvent(this, "Jaxo.vertexType",
                new Integer(JaxoConstants.VERTEXT6),
                new Integer(JaxoConstants.VERTEXT1));
        miscButtons.propertyChange(event);

    }


}
