/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.gui.panel.button;

import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import junit.framework.TestCase;

import net.sf.jaxodraw.JaxoTestUtils;
import net.sf.jaxodraw.util.JaxoConstants;
import net.sf.jaxodraw.util.JaxoLanguage;

/** Tests for JaxoParticleButtons. */
public class JaxoParticleButtonsTest extends TestCase {

    private static final int[] PARTICLE_MODES =
    {
        JaxoConstants.FLINE, JaxoConstants.FLOOP, JaxoConstants.FARC,
        JaxoConstants.FBEZ, JaxoConstants.SLINE, JaxoConstants.SLOOP,
        JaxoConstants.SARC, JaxoConstants.SBEZ, JaxoConstants.GLINE,
        JaxoConstants.GLOOP, JaxoConstants.GARC, JaxoConstants.GBEZ,
        JaxoConstants.PLINE, JaxoConstants.PLOOP, JaxoConstants.PARC,
        JaxoConstants.PBEZ, JaxoConstants.GLLINE, JaxoConstants.GLLOOP,
        JaxoConstants.GLARC, JaxoConstants.GLBEZ
    };
    private static final int NOF_BUTTONS = PARTICLE_MODES.length;
    private static final String[] TOOLTIP_KEYS =
    {
        "Draw_a_fermion_line", "Draw_a_fermion_loop", "Draw_a_fermion_arc",
        "Draw_a_fermion_bezier", "Draw_a_scalar_line", "Draw_a_scalar_loop",
        "Draw_a_scalar_arc", "Draw_a_scalar_bezier", "Draw_a_ghost_line",
        "Draw_a_ghost_loop", "Draw_a_ghost_arc", "Draw_a_ghost_bezier",
        "Draw_a_photon_line", "Draw_a_photon_loop", "Draw_a_photon_arc",
        "Draw_a_photon_bezier", "Draw_a_gluon_line", "Draw_a_gluon_loop",
        "Draw_a_gluon_arc", "Draw_a_gluon_bezier"
    };

    private final JaxoParticleButtons particleButtons = new JaxoParticleButtons();
    private String actionCommand;

    /** Constructor: just calls super().
     * @param testName The name of this test.
     */
    public JaxoParticleButtonsTest(String testName) {
        super(testName);
    }

    /** Tests constructor. */
    public void testConstructor() {
        for (int i = 0; i < NOF_BUTTONS; i++) {
            String name = JaxoConstants.getModeAsString(PARTICLE_MODES[i]);
            Component component = JaxoTestUtils.getChildNamed(particleButtons, name);
            assertNotNull(component);
            JaxoPanelButton button = (JaxoPanelButton) component;
            assertEquals(JaxoConstants.getModeAsString(PARTICLE_MODES[i]),
                button.getName());
            assertEquals(JaxoLanguage.translate(TOOLTIP_KEYS[i]),
                button.getToolTipText());
        }
    }

    /** Tests pressButton. */
    public void testPressButton() {
        for (int i = 0; i < NOF_BUTTONS; i++) {
            particleButtons.pressButton(PARTICLE_MODES[i]);
            for (int j = 0; j < NOF_BUTTONS; j++) {
                String name = JaxoConstants.getModeAsString(PARTICLE_MODES[j]);
                JaxoPanelButton button =
                    (JaxoPanelButton) JaxoTestUtils.getChildNamed(particleButtons, name);
                if (i == j) {
                    assertTrue("should be selected: " + name,
                        button.isSelected());
                } else {
                    assertFalse("should NOT be selected: " + name,
                        button.isSelected());
                }
            }
        }

        particleButtons.pressButton(JaxoConstants.COLOR);
        for (int i = 0; i < NOF_BUTTONS; i++) {
            String name = JaxoConstants.getModeAsString(PARTICLE_MODES[i]);
            JaxoPanelButton button =
                (JaxoPanelButton) JaxoTestUtils.getChildNamed(particleButtons, name);
            assertFalse("should NOT be selected: " + name, button.isSelected());
        }
    }

    /** Tests setDefault. */
    public void testSetDefault() {
        for (int i = 0; i < NOF_BUTTONS; i++) {
            particleButtons.setDefault(PARTICLE_MODES[i]);
            for (int j = 0; j < NOF_BUTTONS; j++) {
                String name = JaxoConstants.getModeAsString(PARTICLE_MODES[j]);
                JaxoPanelButton button =
                    (JaxoPanelButton) JaxoTestUtils.getChildNamed(particleButtons, name);
                if (i == j) {
                    assertTrue("should be default: " + name,
                        button.isDefault());
                } else {
                    assertFalse("should NOT be default: " + name,
                        button.isDefault());
                }
            }
        }

        particleButtons.setDefault(JaxoConstants.COLOR);
        for (int i = 0; i < NOF_BUTTONS; i++) {
            String name = JaxoConstants.getModeAsString(PARTICLE_MODES[i]);
            JaxoPanelButton button =
                (JaxoPanelButton) JaxoTestUtils.getChildNamed(particleButtons, name);
            assertFalse("should NOT be selected: " + name, button.isDefault());
        }
    }

    /** Tests updateLanguage. */
    public void testUpdateLanguage() {
        particleButtons.updateLanguage();
        for (int i = 0; i < NOF_BUTTONS; i++) {
            String name = JaxoConstants.getModeAsString(PARTICLE_MODES[i]);
            JaxoPanelButton button =
                (JaxoPanelButton) JaxoTestUtils.getChildNamed(particleButtons, name);
            assertEquals(JaxoLanguage.translate(TOOLTIP_KEYS[i]),
                button.getToolTipText());
        }
    }

    /** Tests addActionListener. */
    public void testAddActionListener() {
        particleButtons.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent evt) {
                actionCommand = evt.getActionCommand();
            }
        });

        for (int i = 0; i < NOF_BUTTONS; i++) {
            String name = JaxoConstants.getModeAsString(PARTICLE_MODES[i]);
            JaxoPanelButton button =
                (JaxoPanelButton) JaxoTestUtils.getChildNamed(particleButtons, name);
            actionCommand = null;
            button.doClick();
            assertEquals("wrong action command!", name, actionCommand);
        }
    }

}
