/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.io.exports;

import java.awt.Rectangle;

import net.sf.jaxodraw.AbstractJaxoBaseTest;
import net.sf.jaxodraw.graph.JaxoGraph;
import net.sf.jaxodraw.io.JaxoIO;
import net.sf.jaxodraw.object.JaxoList;
import net.sf.jaxodraw.object.JaxoObject;
import net.sf.jaxodraw.object.group.JaxoGroup;
import net.sf.jaxodraw.util.JaxoGeometry;


/**
 * Base Test for JaxoExports.
 */
public abstract class AbstractJaxoExportTest extends AbstractJaxoBaseTest {
    private static final String BASE = "io/export/";
    private JaxoGraph testGraph;

    /** Constructor.
     * @param testName The name of this test.
     */
    protected AbstractJaxoExportTest(String testName) {
        super(testName);
    }

    /** Returns the base directory for test output.
     * @return the base directory for test output.
     */
    protected String getBase() {
        return BASE;
    }

    /** Returns a test graph.
     * @return a test graph.
     */
    protected JaxoGraph getTestGraph() {
        if (testGraph == null) {
            JaxoIO io = new JaxoIO(null);
            testGraph = io.open(getResourceFile("io/", "all-2.0.xml"), false);
        }
        return testGraph;
    }

    /**
     * Return a group that can be used for testing.
     * @return a JaxoGroup.
     */
    protected JaxoGroup getTestGroup() {
        JaxoGroup initGroup = new JaxoGroup();
        JaxoList<JaxoObject> groupObjects = getTestGraph().getObjectList();
        initGroup.setObjectList(groupObjects);
        Rectangle groupBounds = JaxoGeometry.getBounds(groupObjects);
        initGroup.setX(groupBounds.x);
        initGroup.setY(groupBounds.y);
        initGroup.setX2(groupBounds.x + groupBounds.width);
        initGroup.setY2(groupBounds.y + groupBounds.height);
        return initGroup;
    }

    /** Check that the export graph doesn't contain groups. */
    public void testSetGraph() {
        JaxoGraph exportGraph = new JaxoGraph();
        exportGraph.addObject(getTestGroup());
        assertTrue(exportGraph.containsGroups());

        JaxoExport export = getInstance();
        export.setGraph(exportGraph);
        assertFalse(export.getGraph().containsGroups());
        // again, to check that the initial graph has not been modified:
        assertTrue(exportGraph.containsGroups());
    }

    /** All export tests have to test exportTo.
     * @throws Exception Exception.
     */
    public abstract void testExportTo() throws Exception;

    /**
     * Return an instance of a JaxoExport.
     * Used to test methods in the abstract base class.
     * @return an instance of a JaxoExport.
     */
    public abstract JaxoExport getInstance();
}
