/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.object;

import net.sf.jaxodraw.object.arrow.JaxoArrow;
import net.sf.jaxodraw.object.arrow.AbstractBaseArrow;

/**
 * Base Test for JaxoParticleObjects.
 */
public abstract class AbstractJaxoParticleObjectTest
    extends AbstractJaxoExtendedObjectTest {

    private JaxoParticleObject instance;

    /**
     * Constructor.
     *
     * @param testName The name of this test.
     */
    protected AbstractJaxoParticleObjectTest(String testName) {
        super(testName);
    }

    /** JUnit setup: get a new instance of the object. */
    @Override
    protected void setUp() {
        super.setUp();
        this.instance = (JaxoParticleObject) getInstance();
    }

    /** Tests float compare methods. */
    public void testFloatCompare() {
        instance.setStrokeWidth(3.f);
        instance.setDash(1.7f);
        instance.setDLSeparation(2.9f);
        instance.setArrowPosition(0.7f);

        assertTrue(instance.strokeIs(3.000000001f));
        assertTrue(instance.dashIs(1.699999999f));
        assertTrue(instance.dlSepIs(2.90000000002f));
        assertTrue(instance.arrowPositionIs(0.70000000002f));

        assertFalse(instance.strokeIs(3.01f));
        assertFalse(instance.dashIs(1.6f));
        assertFalse(instance.dlSepIs(2.92f));
        assertFalse(instance.arrowPositionIs(0.71f));
    }

    /** Tests setArrowPosition. */
    public void testArrowPosition() {
        try {
            instance.setArrowPosition(-0.5f);
            fail("negative arrow position");
        } catch (IllegalArgumentException e) {
            assertNotNull(e);
            instance.setArrowPosition(-0.00001f);
            assertTrue(instance.arrowPositionIs(0.f));
        }

        try {
            instance.setArrowPosition(3.5f);
            fail("arrow position > 1");
        } catch (IllegalArgumentException e) {
            assertNotNull(e);
            instance.setArrowPosition(1.00001f);
            assertTrue(instance.arrowPositionIs(1.f));
        }
    }

    /**
     * Assert that the given arrow has correct parameters.
     *
     * @param message a message to display in case of a failure.
     * @param testArrow the JaxoArrow to test.
     * @param len the expected arrow length.
     * @param wid the expected arrow width.
     * @param inset the expected arrow inset.
     */
    protected void assertArrow(String message, JaxoArrow testArrow,
            float len, float wid, float inset) {

        AbstractBaseArrow arrow = (AbstractBaseArrow) testArrow;

        assertEquals(message, len, arrow.getArrowLength());
        assertEquals(message, wid, arrow.getArrowWidth());
        assertEquals(message, inset, arrow.getArrowInset());
    }

    /** {@inheritDoc} */
    public String[] serializableProperties() {
        return new String[] {
            "DLSeparation", "amp", "arrow", "arrowPosition", "color",
            "dash", "doubleLine", "flip", "paintArrow", "pointCount",
            "points", "strokeWidth", "wiggles"
        }; // 13
    }

    /**
     * Tests the arrow coordinates of this ParticleObject.
     */
    public abstract void testArrowCoordinates();
}
