/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.object.arc;

import java.awt.geom.Point2D;

import net.sf.jaxodraw.object.AbstractJaxo3PointObjectTest;
import net.sf.jaxodraw.object.arrow.JaxoArrow;


/**
 * Base Test for JaxoArcObjects.
 */
public abstract class AbstractJaxoArcObjectTest
    extends AbstractJaxo3PointObjectTest {

    private JaxoArcObject instance;

    /**
     * Constructor.
     *
     * @param testName The name of this test.
     */
    protected AbstractJaxoArcObjectTest(String testName) {
        super(testName);
    }

    /** JUnit setup: get a new instance of the object. */
    @Override
    protected void setUp() {
        super.setUp();
        this.instance = (JaxoArcObject) getInstance();
    }

      //
     // tests
    //

    /**
     * Test getOpeningAngle.
     */
    public void testOpeningAngle() {
        instance.setLocation(50, 50, 100, 100, 150, 50);
        assertEquals((float) -Math.PI, (float) instance.getOpeningAngle());

        instance.setLocation(150, 50, 100, 100, 50, 50);
        assertEquals((float) Math.PI, (float) instance.getOpeningAngle());

        instance.setLocation(50, 50, 123, 10, 117, 23);
        assertEquals(5.0089f, (float) instance.getOpeningAngle());
    }

    /**
     * Test getEquidistantPoint.
     */
    public void testEquidistantPoint() {
        instance.setLocation(50, 50, 100, 100, 150, 50);
        assertFalse(instance.isClockwise());
        Point2D.Double point = instance.getEquidistantPoint(1, 2);
        assertEquals(100.f, (float) point.x);
        assertEquals(100.f, (float) point.y);

        instance.setLocation(50, 50, 100, 0, 150, 50);
        assertTrue(instance.isClockwise());
        point = instance.getEquidistantPoint(15, 224);
        assertEquals(51.10236f, (float) point.x);
        assertEquals(39.5587f, (float) point.y);
    }

    /**
     * Test getArcPoint.
     */
    public void testArcPoint() {
        instance.setLocation(50, 50, 100, 100, 150, 50);
        Point2D.Double point = instance.getArcPoint(0.f);
        assertEquals(50.f, (float) point.x);
        assertEquals(50.f, (float) point.y);

        point = instance.getArcPoint(0.5f);
        assertEquals(100.f, (float) point.x);
        assertEquals(100.f, (float) point.y);

        point = instance.getArcPoint(1.f);
        assertEquals(150.f, (float) point.x);
        assertEquals(50.f, (float) point.y);

        instance.setLocation(50, 50, 100, 0, 150, 50);
        point = instance.getArcPoint(0.5f);
        assertEquals(100.f, (float) point.x);
        assertEquals(0.f, (float) point.y);
    }

    /** {@inheritDoc} */
    public void testArrowCoordinates() {
        instance.setLocation(50, 50, 100, 0, 150, 50);

        instance.setArrowPosition(0.f);
        JaxoArrow.Coordinates arrow = instance.arrowCoordinates();
        assertEquals(50, (int) Math.round(arrow.getX()));
        assertEquals(50, (int) Math.round(arrow.getY()));
        assertEquals((float) (3.f * Math.PI / 2.f), (float) arrow.getAngle());

        instance.setArrowPosition(0.5f);
        arrow = instance.arrowCoordinates();
        assertEquals(100, (int) Math.round(arrow.getX()));
        assertEquals(0, (int) Math.round(arrow.getY()));
        assertEquals(0.f, (float) arrow.getAngle());

        instance.setArrowPosition(1.f);
        arrow = instance.arrowCoordinates();
        assertEquals(150, (int) Math.round(arrow.getX()));
        assertEquals(50, (int) Math.round(arrow.getY()));
        assertEquals((float) (Math.PI / 2.f), (float) arrow.getAngle());
    }

      //
     // abstract
    //

    /** Tests serialization. */
    public abstract void testSerialization();

}
