/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.object.arc;

import java.awt.Dimension;
import java.awt.Point;
import java.awt.Rectangle;

import net.sf.jaxodraw.JaxoTestUtils;
import net.sf.jaxodraw.io.imports.JaxoAxodraw;
import net.sf.jaxodraw.object.JaxoObject;
import net.sf.jaxodraw.object.JaxoObjectFactory;
import net.sf.jaxodraw.object.loop.JaxoFLoop;
import net.sf.jaxodraw.util.JaxoColor;
import net.sf.jaxodraw.util.JaxoConstants;
import net.sf.jaxodraw.util.JaxoPrefs;


/** Tests JaxoGlArc. */
public class JaxoGlArcTest extends AbstractJaxoArcObjectTest {
    private JaxoGlArc arc = new JaxoGlArc();

    /** Constructor: just calls super().
     * @param testName The name of this test.
     */
    public JaxoGlArcTest(String testName) {
        super(testName);
    }

    /** JUnit setup: initialize the object. */
    @Override
    protected void setUp() {
        super.setUp();
        init();
    }

    private void init() {
        arc.setLocation(50, 50, 100, 50, 100, 100);
        arc.setColor(JaxoColor.GREEN);
        arc.setStrokeWidth(1.f);
        arc.setAmp(15);
        arc.setWiggles(15);
        arc.setPaintArrow(false);
        arc.setFlip(false);
        arc.setDoubleLine(false);
        arc.setDLSeparation(2.f);
    }

    /** {@inheritDoc} */
    public JaxoObject getInstance() {
        return arc;
    }

    /** Tests serialization. */
    public void testSerialization() {
        arc.setStrokeWidth(1.5f);
        arc.setAmp(35);
        arc.setWiggles(25);
        arc.setPaintArrow(true);
        arc.setFlip(true);
        JaxoTestUtils.checkSerializedEquality(arc);
    }

    /** Tests that a copy of an object is still the same object. */
    public void testCopy() {
        JaxoGlArc arc2 = new JaxoGlArc();
        arc2.setState(arc);

        arc.setStrokeWidth(1.5f);
        arc.setAmp(35);
        arc.setWiggles(25);
        arc.setPaintArrow(true);
        arc.setFlip(true);
        JaxoTestUtils.checkCopyEquality(arc);

        assertFalse(arc2.isCopy(arc));

        arc2.setStrokeWidth(1.5f);
        arc2.setAmp(35);
        arc2.setWiggles(25);
        arc2.setPaintArrow(true);
        arc2.setFlip(true);
        assertTrue(arc2.isCopy(arc));

        JaxoFLoop loop = new JaxoFLoop();
        assertFalse(arc.isCopy(loop));

    }

    /** Tests identity. */
    public void testIdentity() {
        JaxoGlArc arc2 = arc;
        assertEquals(arc, arc2);
    }

    /** Tests the latex command. */
    public void testLatexCommand() {
        Dimension dim = new Dimension(74, 74);

        String expected = "\\GluonArc[clock](75,-1)(35.355,135,-45){7.5}{15}";
        String actual = arc.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

        arc.setDoubleLine(true);
        arc.setDLSeparation(3.f);

        dim.setSize(81, 81);
        expected = "\\GluonArc[double,sep=3,clock](75,6)(35.355,135,-45){7.5}{15}";
        actual = arc.latexCommand(1.f, dim);
        assertEquals("Latex double line commands do not match!", expected,
            actual);
    }

    /** Tests axodraw. */
    public void testAxodraw() {
        JaxoAxodraw axo = new JaxoAxodraw();
        JaxoObject ob = axo.getJaxoObject("\\SetWidth{1.0}");
        assertNull(ob);
        String command;

        command = "\\GlueArc(75,-1)(35.36,135,-45){-7.5}{25}";
        arc = (JaxoGlArc) axo.getJaxoObject(command);
        checkArcParameters(false);

        command =
            "\\GlueArc(75,6)(38.36,135,-45){-7.5}{25}\\GlueArc(75,6)(32.36,135,-45){-7.5}{25}%%JaxoDrawID:DoubleLine(3)";
        arc = (JaxoGlArc) axo.getJaxoObject(command);
        checkArcParameters(true);

    }

    private void checkArcParameters(boolean doubleLine) {
        assertEquals("GlArc doesn't have the right color!", JaxoColor.BLACK,
            arc.getColor());
        assertTrue("GlArc doesn't have the right stroke!", arc.strokeIs(1.f));
        assertEquals("Mismatch in GlArc doubleLine!", doubleLine,
            arc.isDoubleLine());

        float dlsep = 2.f;
        if (doubleLine) {
            dlsep = 3.f;
        }

        assertTrue("Mismatch in GlArc doubleLine separation! Got: "
            + arc.getDLSeparation(), arc.dlSepIs(dlsep));
        assertEquals("Mismatch in GlArc amp!", 15, arc.getAmp());
        assertEquals("Mismatch in GlArc wiggles!", 25, arc.getWiggles());
    }

    /** Tests the getBounds() method. */
    public void testGetBounds() {
        Rectangle bBox = arc.getBounds();
        Rectangle expected = new Rectangle(46, 31, 73, 73);
        assertEquals("Bounding box not correct!", expected, bBox);

        arc.setLocation(50, 50, 100, 100, 150, 50);
        arc.setDoubleLine(true);
        bBox = arc.getBounds();
        expected.setRect(41, 49, 118, 61);
        assertEquals("DL Bounding box not correct!", expected, bBox);

        assertNotNull(arc.getBounds2D());
    }

    /** Tests createNewObject in JaxoObjectFactory. */
    public void testCreateNewObject() {
        Point p = new Point(50, 50);
        Point q = new Point(100, 50);
        Point r = new Point(100, 100);
        JaxoObject ob =
            JaxoObjectFactory.newObject(JaxoConstants.GLARC, new Point[] {p, q, r});

        assertTrue("createNewObject did not return a GlArc!",
            (ob instanceof JaxoGlArc));

        arc = (JaxoGlArc) ob;
        assertEquals("GlArc doesn't have the right x1!", 50, arc.getX());
        assertEquals("GlArc doesn't have the right y1!", 50, arc.getY());
        assertEquals("GlArc doesn't have the right x2!", 100, arc.getX2());
        assertEquals("GlArc doesn't have the right y2!", 50, arc.getY2());
        assertEquals("GlArc doesn't have the right x3!", 100, arc.getX3());
        assertEquals("GlArc doesn't have the right y3!", 100, arc.getY3());
        assertEquals("GlArc doesn't have the right color!",
            JaxoColor.getColor(JaxoPrefs.getStringPref(JaxoPrefs.PREF_COLOR)),
            arc.getColor());
        assertTrue("GlArc doesn't have the right stroke!",
            arc.strokeIs(JaxoPrefs.getFloatPref(JaxoPrefs.PREF_LINEWIDTH)));
        assertEquals("GlArc doesn't have the right double line!", false,
            arc.isDoubleLine());
        assertTrue("GlArc doesn't have the right dlsep!",
            arc.dlSepIs(JaxoPrefs.getFloatPref(JaxoPrefs.PREF_DLSEP)));
        assertEquals("GlArc doesn't have the right amplitude!",
            JaxoPrefs.getIntPref(JaxoPrefs.PREF_AMPLITUDE), arc.getAmp());
    }

    /** {@inheritDoc} */
    public String className() {
        return "JaxoGlArc";
    }

    /** {@inheritDoc} */
    public void testJaxo20Compat() {
        JaxoGlArc object =
                (JaxoGlArc) instanceFrom20OpenGraph(JaxoGlArc.class);
        assertNotNull(object);
        check20Object("opened wrong object!", object);

        object = (JaxoGlArc) instanceFrom20ImportGraph(JaxoGlArc.class);
        assertNotNull(object);
        check20Object("imported wrong object!", object);
    }

    private void check20Object(String message, JaxoGlArc object) {

        assertEquals(message, 314, object.getX());
        assertEquals(message, 440, object.getY());
        assertEquals(message, 370, object.getX2());
        assertEquals(message, 446, object.getY2());
        assertEquals(message, 364, object.getX3());
        assertEquals(message, 502, object.getY3());

        assertTrue(message, object.strokeIs(1.4f));
        assertEquals(message, 17, object.getAmp());
        assertEquals(message, 7, object.getWiggles());

        assertFalse(message, object.isDoubleLine());

        assertEquals(message, JaxoColor.BLUE_GREEN, object.getColor());
    }
}
