/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.object.arc;

import java.awt.Dimension;
import java.awt.Point;
import java.awt.Rectangle;

import net.sf.jaxodraw.JaxoTestUtils;
import net.sf.jaxodraw.io.imports.JaxoAxodraw;
import net.sf.jaxodraw.object.JaxoObject;
import net.sf.jaxodraw.object.JaxoObjectFactory;
import net.sf.jaxodraw.object.loop.JaxoFLoop;
import net.sf.jaxodraw.util.JaxoColor;
import net.sf.jaxodraw.util.JaxoConstants;
import net.sf.jaxodraw.util.JaxoPrefs;


/** Tests JaxoPArc. */
public class JaxoPArcTest extends AbstractJaxoArcObjectTest {
    private JaxoPArc arc = new JaxoPArc();

    /** Constructor: just calls super().
     * @param testName The name of this test.
     */
    public JaxoPArcTest(String testName) {
        super(testName);
    }

    /** JUnit setup: initialize the object. */
    @Override
    protected void setUp() {
        super.setUp();
        init();
    }

    private void init() {
        arc.setLocation(50, 50, 100, 50, 100, 100);
        arc.setColor(JaxoColor.GREEN);
        arc.setStrokeWidth(1.f);
        arc.setAmp(-15);
        arc.setWiggles(13);
        arc.setPaintArrow(false);
        arc.setFlip(false);
        arc.setDoubleLine(false);
        arc.setDLSeparation(2.f);
    }

    /** {@inheritDoc} */
    public JaxoObject getInstance() {
        return arc;
    }

    /** Tests serialization. */
    public void testSerialization() {
        arc.setStrokeWidth(1.5f);
        arc.setAmp(35);
        arc.setPaintArrow(true);
        arc.setFlip(true);
        JaxoTestUtils.checkSerializedEquality(arc);
    }

    /** Tests that a copy of an object is still the same object. */
    public void testCopy() {
        JaxoPArc arc2 = new JaxoPArc();
        arc2.setState(arc);

        arc.setStrokeWidth(1.5f);
        arc.setAmp(35);
        arc.setPaintArrow(true);
        arc.setFlip(true);
        JaxoTestUtils.checkCopyEquality(arc);

        assertFalse(arc2.isCopy(arc));

        arc2.setStrokeWidth(1.5f);
        arc2.setAmp(35);
        arc2.setPaintArrow(true);
        arc2.setFlip(true);
        assertTrue(arc2.isCopy(arc));

        JaxoFLoop loop = new JaxoFLoop();
        assertFalse(arc.isCopy(loop));

    }

    /** Tests identity. */
    public void testIdentity() {
        JaxoPArc arc2 = arc;
        assertEquals(arc, arc2);
    }

    /** Tests the latex command. */
    public void testLatexCommand() {
        Dimension dim = new Dimension(68, 68);

        String expected = "\\PhotonArc[clock](75,-7)(35.355,135,-45){-7.5}{13.5}";
        String actual = arc.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

        arc.setDoubleLine(true);
        arc.setDLSeparation(3.f);

        dim.setSize(74, 74);
        expected = "\\PhotonArc[double,sep=3,clock](75,-1)(35.355,135,-45){-7.5}{13.5}";
        actual = arc.latexCommand(1.f, dim);
        assertEquals("Latex double line commands do not match!", expected,
            actual);

    }

    /** Tests axodraw. */
    public void testAxodraw() {
        JaxoAxodraw axo = new JaxoAxodraw();
        JaxoObject ob = axo.getJaxoObject("\\SetWidth{1.0}");
        assertNull(ob);
        String command;

        command = "\\PhotonArc(75,-7)(35.36,-45,135){-7.5}{13.5}";
        arc = (JaxoPArc) axo.getJaxoObject(command);
        checkArcParameters(false);

        command =
            "\\PhotonArc(75,-1)(38.36,-45,135){-7.5}{13.5}\\PhotonArc(75,-1)(32.36,-45,135){-7.5}{13.5}%%JaxoDrawID:DoubleLine(3)";
        arc = (JaxoPArc) axo.getJaxoObject(command);
        checkArcParameters(true);

    }

    private void checkArcParameters(boolean doubleLine) {
        assertEquals("PArc doesn't have the right color!", JaxoColor.BLACK,
            arc.getColor());
        assertTrue("PArc doesn't have the right stroke!", arc.strokeIs(1.f));
        assertEquals("Mismatch in PArc doubleLine!", doubleLine,
            arc.isDoubleLine());

        float dlsep = 2.f;
        if (doubleLine) {
            dlsep = 3.f;
        }

        assertTrue("Mismatch in PArc doubleLine separation! Got: "
            + arc.getDLSeparation(), arc.dlSepIs(dlsep));
        assertEquals("Mismatch in PArc amp!", -15, arc.getAmp());
        assertEquals("Mismatch in PArc wiggles!", 13, arc.getWiggles());
    }

    /** Tests the getBounds() method. */
    public void testGetBounds() {
        Rectangle bBox = arc.getBounds();
        Rectangle expected = new Rectangle(49, 31, 70, 70);
        assertEquals("Bounding box not correct!", expected, bBox);

        arc.setLocation(50, 50, 100, 100, 150, 50);
        arc.setDoubleLine(true);
        bBox = arc.getBounds();
        expected.setRect(41, 49, 118, 60);
        assertEquals("DL Bounding box not correct!", expected, bBox);

        assertNotNull(arc.getBounds2D());
    }

    /** Tests createNewObject in JaxoObjectFactory. */
    public void testCreateNewObject() {
        Point p = new Point(50, 50);
        Point q = new Point(100, 50);
        Point r = new Point(100, 100);
        JaxoObject ob =
            JaxoObjectFactory.newObject(JaxoConstants.PARC, new Point[] {p, q, r});

        assertTrue("createNewObject did not return a PArc!",
            (ob instanceof JaxoPArc));

        arc = (JaxoPArc) ob;
        assertEquals("PArc doesn't have the right x1!", 50, arc.getX());
        assertEquals("PArc doesn't have the right y1!", 50, arc.getY());
        assertEquals("PArc doesn't have the right x2!", 100, arc.getX2());
        assertEquals("PArc doesn't have the right y2!", 50, arc.getY2());
        assertEquals("PArc doesn't have the right x3!", 100, arc.getX3());
        assertEquals("PArc doesn't have the right y3!", 100, arc.getY3());
        assertEquals("PArc doesn't have the right color!",
            JaxoColor.getColor(JaxoPrefs.getStringPref(JaxoPrefs.PREF_COLOR)),
            arc.getColor());
        assertTrue("PArc doesn't have the right stroke!",
            arc.strokeIs(JaxoPrefs.getFloatPref(JaxoPrefs.PREF_LINEWIDTH)));
        assertEquals("PArc doesn't have the right double line!", false,
            arc.isDoubleLine());
        assertTrue("PArc doesn't have the right dlsep!",
            arc.dlSepIs(JaxoPrefs.getFloatPref(JaxoPrefs.PREF_DLSEP)));
        assertEquals("PArc doesn't have the right amplitude!",
            JaxoPrefs.getIntPref(JaxoPrefs.PREF_AMPLITUDE), arc.getAmp());
    }

    /** {@inheritDoc} */
    public String className() {
        return "JaxoPArc";
    }

    /** {@inheritDoc} */
    public void testJaxo20Compat() {
        JaxoPArc object =
                (JaxoPArc) instanceFrom20OpenGraph(JaxoPArc.class);
        assertNotNull(object);
        check20Object("opened wrong object!", object);

        object = (JaxoPArc) instanceFrom20ImportGraph(JaxoPArc.class);
        assertNotNull(object);
        check20Object("imported wrong object!", object);
    }

    private void check20Object(String message, JaxoPArc object) {
        assertEquals(message, 364, object.getX());
        assertEquals(message, 383, object.getY());
        assertEquals(message, 373, object.getX2());
        assertEquals(message, 322, object.getY2());
        assertEquals(message, 312, object.getX3());
        assertEquals(message, 313, object.getY3());

        assertTrue(message, object.strokeIs(1.2f));
        assertEquals(message, -10, object.getAmp());
        assertEquals(message, 5, object.getWiggles());
        assertFalse(message, object.isSymmetric());

        assertFalse(message, object.isDoubleLine());

        assertEquals(message, JaxoColor.WILDSTRAWBERRY, object.getColor());
    }

    /** {@inheritDoc} */
    @Override
    public String[] serializableProperties() {
        return new String[] {
            "DLSeparation", "amp", "arrow", "arrowPosition", "color",
            "dash", "doubleLine", "flip", "paintArrow", "pointCount",
            "points", "strokeWidth", "wiggles", "symmetric"
        }; // 14
    }
}
