/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.object.bezier;

import java.awt.Dimension;
import java.awt.Point;
import java.awt.Rectangle;

import net.sf.jaxodraw.JaxoTestUtils;
import net.sf.jaxodraw.io.imports.JaxoAxodraw4J;
import net.sf.jaxodraw.object.JaxoObject;
import net.sf.jaxodraw.object.JaxoObjectFactory;
import net.sf.jaxodraw.object.arc.JaxoGArc;
import net.sf.jaxodraw.object.loop.JaxoFLoop;
import net.sf.jaxodraw.util.JaxoColor;
import net.sf.jaxodraw.util.JaxoConstants;
import net.sf.jaxodraw.util.JaxoPrefs;


/** Tests JaxoGlBezier. */
public class JaxoGlBezierTest extends AbstractJaxoBezierObjectTest {
    private JaxoGlBezier bezier = new JaxoGlBezier();

    /** Constructor: just calls super().
     * @param testName The name of this test.
     */
    public JaxoGlBezierTest(String testName) {
        super(testName);
    }

    /** JUnit setup: initialize the object. */
    @Override
    protected void setUp() {
        super.setUp();
        init();
    }

    private void init() {
        bezier.setLocation(50, 50, 100, 50, 100, 100, 50, 100);
        bezier.setColor(JaxoColor.GREEN);
        bezier.setStrokeWidth(1.f);
        bezier.setAmp(15);
        bezier.setWiggles(15);
        bezier.setDoubleLine(false);
        bezier.setDLSeparation(2.f);
    }


    /** {@inheritDoc} */
    public JaxoObject getInstance() {
        return bezier;
    }

    /** Tests serialization. */
    public void testSerialization() {
        bezier.setStrokeWidth(1.5f);
        bezier.setAmp(35);
        bezier.setWiggles(25);
        JaxoTestUtils.checkSerializedEquality(bezier);
    }

    /** Tests that a copy of an object is still the same object. */
    public void testCopy() {
        bezier.setStrokeWidth(1.5f);
        bezier.setAmp(35);
        bezier.setWiggles(25);
        JaxoTestUtils.checkCopyEquality(bezier);

        JaxoGlBezier bezier2 = new JaxoGlBezier();
        bezier2.setLocation(50, 50, 100, 50, 100, 100, 50, 100);
        bezier2.setColor(JaxoColor.GREEN);
        bezier2.setStrokeWidth(1.f);
        bezier2.setAmp(35);
        bezier2.setWiggles(25);
        bezier2.setDoubleLine(false);
        bezier2.setDLSeparation(2.f);
        assertFalse(bezier2.isCopy(bezier));

        bezier2.setStrokeWidth(1.5f);
        assertTrue(bezier2.isCopy(bezier));

        JaxoFLoop loop = new JaxoFLoop();
        assertFalse(bezier.isCopy(loop));

        JaxoGlBezier gbezier = new JaxoGlBezier();
        JaxoGArc garc = new JaxoGArc();
        assertFalse("GlBezier is copy of GArc!", gbezier.isCopy(garc));

    }

    /** Tests the latex command. */
    public void testLatexCommand() {
        Dimension dim = new Dimension(100, 100);

        String expected = "%JaxoID:GlBez(50,50)(100,50)(100,100)(50,100){15}{15}[true]";
        String actual = bezier.latexCommand(1.f, dim);
        assertTrue("GlBezier Latex commands do not match!", actual.endsWith(expected));
    }

    /** Tests axodraw. */
    public void testAxodraw() {
        JaxoAxodraw4J axo = new JaxoAxodraw4J();
        JaxoObject ob = axo.getJaxoObject("\\SetWidth{1.0}");
        assertNull(ob);
        String command;

        command = "\\Bezier blahblah%JaxoID:GlBez(50,50)(100,50)(100,100)(50,100){15}{15}[true]";
        bezier = (JaxoGlBezier) axo.getJaxoObject(command);
        assertNotNull("GlBezier not recognized!", bezier);

        assertEquals("GlBezier doesn't have the right color!", JaxoColor.BLACK,
            bezier.getColor());
        assertTrue("GlBezier doesn't have the right stroke!",
            Math.abs(bezier.getStrokeWidth() - 1.f) < 0.0001f);
        assertFalse("GlBezier has an arrow!", bezier.isPaintArrow());

        assertEquals("GlBezier doesn't have the right x1!", 50, bezier.getX());
        assertEquals("GlBezier doesn't have the right y1!", 50, bezier.getY());
        assertEquals("GlBezier doesn't have the right x2!", 100, bezier.getX2());
        assertEquals("GlBezier doesn't have the right y2!", 50, bezier.getY2());
        assertEquals("GlBezier doesn't have the right x3!", 100, bezier.getX3());
        assertEquals("GlBezier doesn't have the right y3!", 100, bezier.getY3());
        assertEquals("GlBezier doesn't have the right x4!", 50, bezier.getX4());
        assertEquals("GlBezier doesn't have the right y4!", 100, bezier.getY4());
    }

    /** Tests the getBounds() method. */
    public void testGetBounds() {
        Rectangle bBox = bezier.getBounds();
        Rectangle expected = new Rectangle(49, 42, 44, 66);
        assertEquals("Bounding box not correct!", expected, bBox);

        assertNotNull(bezier.getBounds2D());
    }

    /** Tests createNewObject in JaxoObjectFactory. */
    public void testCreateNewObject() {
        Point p = new Point(50, 50);
        Point q = new Point(100, 50);
        Point r = new Point(100, 100);
        Point s = new Point(50, 100);
        JaxoObject ob =
            JaxoObjectFactory.newObject(JaxoConstants.GLBEZ, new Point[] {p, q, r, s});

        assertTrue("createNewObject did not return a GlBezier!",
            (ob instanceof JaxoGlBezier));

        bezier = (JaxoGlBezier) ob;
        assertEquals("GlBezier doesn't have the right x1!", 50, bezier.getX());
        assertEquals("GlBezier doesn't have the right y1!", 50, bezier.getY());
        assertEquals("GlBezier doesn't have the right x2!", 100, bezier.getX2());
        assertEquals("GlBezier doesn't have the right y2!", 50, bezier.getY2());
        assertEquals("GlBezier doesn't have the right x3!", 100, bezier.getX3());
        assertEquals("GlBezier doesn't have the right y3!", 100, bezier.getY3());
        assertEquals("GlBezier doesn't have the right x4!", 50, bezier.getX4());
        assertEquals("GlBezier doesn't have the right y4!", 100, bezier.getY4());
        assertEquals("GlBezier doesn't have the right color!",
            JaxoColor.getColor(JaxoPrefs.getStringPref(JaxoPrefs.PREF_COLOR)),
            bezier.getColor());
        assertTrue("GlBezier doesn't have the right stroke!",
            bezier.strokeIs(JaxoPrefs.getFloatPref(JaxoPrefs.PREF_LINEWIDTH)));
        assertEquals("GlBezier doesn't have the right double line!", false,
            bezier.isDoubleLine());
        assertTrue("GlBezier doesn't have the right dlsep!",
            bezier.dlSepIs(JaxoPrefs.getFloatPref(JaxoPrefs.PREF_DLSEP)));
    }
    /** {@inheritDoc} */
    public String className() {
        return "JaxoGlBezier";
    }

    /** {@inheritDoc} */
    public void testJaxo20Compat() {
        JaxoGlBezier object =
                (JaxoGlBezier) instanceFrom20OpenGraph(JaxoGlBezier.class);
        assertNotNull(object);
        check20Object("opened wrong object!", object);

        object =
                (JaxoGlBezier) instanceFrom20ImportGraph(JaxoGlBezier.class);
        assertNotNull(object);
        check20Object("imported wrong object!", object);
    }

    private void check20Object(String message, JaxoGlBezier object) {
        assertEquals(message, 585, object.getX());
        assertEquals(message, 393, object.getY());
        assertEquals(message, 677, object.getX2());
        assertEquals(message, 526, object.getY2());
        assertEquals(message, 747, object.getX3());
        assertEquals(message, 499, object.getY3());
        assertEquals(message, 636, object.getX4());
        assertEquals(message, 370, object.getY4());

        assertTrue(message, object.strokeIs(0.8f));

        assertEquals(message, JaxoColor.CERULEAN, object.getColor());

        assertEquals(message, 17, object.getAmp());
        assertEquals(message, 23, object.getWiggles());
        assertTrue(message, object.isNoFreqStretching());
    }
}
