/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.object.line;

import java.awt.Dimension;
import java.awt.Point;
import java.awt.Rectangle;

import net.sf.jaxodraw.JaxoTestUtils;
import net.sf.jaxodraw.io.imports.JaxoAxodraw;
import net.sf.jaxodraw.object.JaxoObject;
import net.sf.jaxodraw.object.JaxoObjectFactory;
import net.sf.jaxodraw.object.loop.JaxoFLoop;
import net.sf.jaxodraw.util.JaxoColor;
import net.sf.jaxodraw.util.JaxoConstants;
import net.sf.jaxodraw.util.JaxoPrefs;


/** Tests JaxoSLine. */
public class JaxoSLineTest extends AbstractJaxoLineObjectTest {
    private JaxoSLine line = new JaxoSLine();

    /** Constructor: just calls super().
     * @param testName The name of this test.
     */
    public JaxoSLineTest(String testName) {
        super(testName);
    }

    /** JUnit setup: initialize the object. */
    @Override
    protected void setUp() {
        super.setUp();
        init();
    }

    private void init() {
        line.setLocation(100, 100);
        line.setRelWAndH(100, 100);
        line.setStrokeWidth(1.f);
        line.setDash(10.f);
        line.setColor(JaxoColor.GREEN);
        line.setPaintArrow(false);
        line.setArrowPosition(0.5f);
        line.setFlip(false);
        line.setDoubleLine(false);
        line.setDLSeparation(2.f);
    }

    /** {@inheritDoc} */
    public JaxoObject getInstance() {
        return line;
    }

    /** Tests serialization. */
    public void testSerialization() {
        line.setPaintArrow(true);
        line.setFlip(true);
        line.setDash(2.5f);
        line.setDoubleLine(true);
        line.setDLSeparation(4.f);
        JaxoTestUtils.checkSerializedEquality(line);
    }

    /** Tests that a copy of an object is still the same object. */
    public void testCopy() {
        JaxoSLine line2 = new JaxoSLine();
        line2.setState(line);

        line.setPaintArrow(true);
        line.setFlip(true);
        line.setDash(7.f);
        line.setDoubleLine(true);
        line.setDLSeparation(4.f);
        JaxoTestUtils.checkCopyEquality(line);

        assertFalse(line2.isCopy(line));

        line2.setPaintArrow(true);
        line2.setFlip(true);
        line2.setDash(7.f);
        line2.setDoubleLine(true);
        line2.setDLSeparation(4.f);
        assertTrue(line2.isCopy(line));

        JaxoFLoop loop = new JaxoFLoop();
        assertFalse(line.isCopy(loop));

    }

    /** Tests identity. */
    public void testIdentity() {
        JaxoSLine line2 = line;
        assertEquals(line, line2);
    }

    /** Tests the latex command. */
    public void testLatexCommand() {
        Dimension dim = new Dimension(100, 100);

        String expected = "\\Line[dash,dashsize=10](100,0)(200,-100)";
        String actual = line.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

        line.setPaintArrow(true);
        expected = "\\Line[dash,dashsize=10,arrow,arrowpos=0.5,arrowlength=5,arrowwidth=2,arrowinset=0.2](100,0)(200,-100)";
        actual = line.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

        line.setFlip(true);
        expected = "\\Line[dash,dashsize=10,arrow,arrowpos=0.5,arrowlength=5,arrowwidth=2,arrowinset=0.2,flip](100,0)(200,-100)";
        actual = line.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

        // reset to initial state
        line.setPaintArrow(false);
        line.setFlip(false);

        line.setDoubleLine(true);
        line.setDLSeparation(4.f);

        expected = "\\Line[dash,dashsize=10,double,sep=4](100,0)(200,-100)";
        actual = line.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

        line.setFlip(true);
        expected = "\\Line[dash,dashsize=10,double,sep=4](100,0)(200,-100)";
        actual = line.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

        line.setRelWAndH(0, 0);
        expected = "%";
        actual = line.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

        // reset to initial state
        line.setPaintArrow(false);
        line.setFlip(false);
        line.setDoubleLine(false);
        line.setRelWAndH(-100, -100);
        expected = "\\Line[dash,dashsize=10](100,0)(0,100)";
        actual = line.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

    }

    /** Tests axodraw. */
    public void testAxodraw() {
        JaxoAxodraw axo = new JaxoAxodraw();
        JaxoObject ob = axo.getJaxoObject("\\SetWidth{1.0}");
        assertNull(ob);
        String command;

        command = "\\DashLine(100,200)(200,300){10}";
        line = (JaxoSLine) axo.getJaxoObject(command);
        checkLineParameters(line, false);

        command = "\\DashArrowLine(100,200)(200,300){10}";
        line = (JaxoSLine) axo.getJaxoObject(command);
        checkLineParameters(line, false);

        command =
            "\\DashLine(102.12,-191.88)(202.12,-291.88){10}\\DashLine(97.88,-196.12)(197.88,-296.12){10}%%JaxoDrawID:DoubleLine(3)";
        line = (JaxoSLine) axo.getJaxoObject(command);
        checkLineParameters(line, true);

        command =
            "\\DashArrowLine(102.12,-191.88)(202.12,-291.88){10}\\DashArrowLine(97.88,-196.12)(197.88,-296.12){10}%%JaxoDrawID:DoubleLine(3)";
        line = (JaxoSLine) axo.getJaxoObject(command);
        checkLineParameters(line, true);

    }

    private void checkLineParameters(JaxoSLine aline, boolean doubleLine) {
        assertEquals("SLine doesn't have the right color!", JaxoColor.BLACK,
            aline.getColor());
        assertTrue("SLine doesn't have the right stroke!",
            Math.abs(aline.getStrokeWidth() - 1.f) < 0.0001f);
        assertTrue("SLine doesn't have the right dash!",
            Math.abs(aline.getDash() - 10.f) < 0.0001f);
        assertEquals("Mismatch in SLine doubleLine!", doubleLine,
            aline.isDoubleLine());

        float dlsep = 2.f;
        int x = 100;
        int y = -200;
        if (doubleLine) {
            dlsep = 3.f;
            x = 100;
            y = 194;
        }

        assertEquals("SLine doesn't have the right x!", x, aline.getX());
        assertEquals("SLine doesn't have the right y!", y, aline.getY());
        assertTrue("Mismatch in SLine doubleLine separation!",
            Math.abs(aline.getDLSeparation() - dlsep) < 0.0001f);

    }

    /** Tests the getBounds() method. */
    public void testGetBounds() {
        Rectangle bBox = line.getBounds();
        Rectangle expected = new Rectangle(99, 99, 102, 102);
        assertEquals("Bounding box not correct!", expected, bBox);
        line.setPaintArrow(true);
        line.setLocation(10, 10, 20, 10);
        bBox = line.getBounds();
        expected.setRect(9, 7, 12, 6);
        assertEquals("Arc Bounding box not correct!", expected, bBox);
        line.setDoubleLine(true);
        bBox = line.getBounds();
        expected.setRect(9, 7, 12, 6);
        assertEquals("DL Bounding box not correct!", expected, bBox);

        assertNotNull(line.getBounds2D());
    }

    /** Tests createNewObject in JaxoObjectFactory. */
    public void testCreateNewObject() {
        Point p = new Point(50, 50);
        JaxoObject ob =
            JaxoObjectFactory.newObject(JaxoConstants.SLINE, new Point[] {p, p});

        assertTrue("createNewObject did not return a SLine!",
            (ob instanceof JaxoSLine));

        line = (JaxoSLine) ob;
        assertEquals("SLine doesn't have the right x!", 50, line.getX());
        assertEquals("SLine doesn't have the right y!", 50, line.getY());
        assertEquals("SLine doesn't have the right color!",
            JaxoColor.getColor(JaxoPrefs.getStringPref(JaxoPrefs.PREF_COLOR)),
            line.getColor());
        assertTrue("SLine doesn't have the right stroke!",
            line.strokeIs(JaxoPrefs.getFloatPref(JaxoPrefs.PREF_LINEWIDTH)));
        assertEquals("SLine doesn't have the right arrow!",
            JaxoPrefs.getBooleanPref(JaxoPrefs.PREF_ARROW), line.isPaintArrow());
        assertEquals("SLine doesn't have the right flip!", false,
            line.isFlip());
        assertEquals("SLine doesn't have the right double line!", false,
            line.isDoubleLine());
        assertTrue("SLine doesn't have the right dlsep!",
            line.dlSepIs(JaxoPrefs.getFloatPref(JaxoPrefs.PREF_DLSEP)));
        assertTrue("SLine doesn't have the right dash!", line.dashIs(10.f));

    }

    /** {@inheritDoc} */
    public String className() {
        return "JaxoSLine";
    }

    /** {@inheritDoc} */
    public void testJaxo20Compat() {
        JaxoSLine object = (JaxoSLine) instanceFrom20OpenGraph(JaxoSLine.class);
        assertNotNull(object);
        check20Object("opened wrong object!", object);

        object = (JaxoSLine) instanceFrom20ImportGraph(JaxoSLine.class);
        assertNotNull(object);
        check20Object("imported wrong object!", object);
    }

    private void check20Object(String message, JaxoSLine object) {
        assertEquals(message, 46, object.getX());
        assertEquals(message, 88, object.getY());
        assertEquals(message, 129, object.getX2());
        assertEquals(message, 151, object.getY2());

        assertFalse(message, object.isDoubleLine());
        assertTrue(message, object.strokeIs(1.4f));
        assertTrue(message, object.dashIs(9.996f));

        assertEquals(message, JaxoColor.LAVENDER, object.getColor());

        assertTrue(message, object.isPaintArrow());
        assertTrue(message, object.isFlip());
        assertEquals(message, 0.f, object.getArrowPosition());
        assertArrow(message, object.getArrow(), 7.5f, 3.f, 0.2f);
    }
}
