/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.object.loop;

import java.awt.Dimension;
import java.awt.Point;
import java.awt.Rectangle;

import net.sf.jaxodraw.JaxoTestUtils;
import net.sf.jaxodraw.io.imports.JaxoAxodraw;
import net.sf.jaxodraw.object.JaxoObject;
import net.sf.jaxodraw.object.JaxoObjectFactory;
import net.sf.jaxodraw.object.arc.JaxoFArc;
import net.sf.jaxodraw.util.JaxoColor;
import net.sf.jaxodraw.util.JaxoConstants;
import net.sf.jaxodraw.util.JaxoPrefs;


/** Tests JaxoFLoop. */
public class JaxoFLoopTest extends AbstractJaxoLoopObjectTest {
    private JaxoFLoop loop = new JaxoFLoop();

    /** Constructor: just calls super().
     * @param testName The name of this test.
     */
    public JaxoFLoopTest(String testName) {
        super(testName);
    }

    /** JUnit setup: initialize the object. */
    @Override
    protected void setUp() {
        super.setUp();
        init();
    }

    private void init() {
        loop.setLocation(100, 100);
        loop.setRelWAndH(35, 37);
        loop.setColor(JaxoColor.RED);
        loop.setStrokeWidth(1.f);
        loop.setPaintArrow(false);
        loop.setFlip(false);
        loop.setDoubleLine(false);
        loop.setDLSeparation(2.f);
    }

    /** {@inheritDoc} */
    public JaxoObject getInstance() {
        return loop;
    }

    /** Tests serialization. */
    public void testSerialization() {
        loop.setStrokeWidth(1.6f);
        loop.setPaintArrow(true);
        loop.setFlip(true);
        loop.setDoubleLine(true);
        JaxoTestUtils.checkSerializedEquality(loop);
    }

    /** Tests that a copy of an object is still the same object. */
    public void testCopy() {
        JaxoFLoop loop2 = new JaxoFLoop();
        loop2.setState(loop);

        loop.setStrokeWidth(1.6f);
        loop.setPaintArrow(true);
        loop.setFlip(true);
        loop.setDoubleLine(true);
        JaxoTestUtils.checkCopyEquality(loop);

        assertFalse(loop2.isCopy(loop));

        loop2.setStrokeWidth(1.6f);
        loop2.setPaintArrow(true);
        loop2.setFlip(true);
        loop2.setDoubleLine(true);
        assertTrue(loop2.isCopy(loop));

        JaxoFArc arc = new JaxoFArc();
        assertFalse(loop.isCopy(arc));

    }

    /** Tests identity. */
    public void testIdentity() {
        JaxoFLoop loop2 = loop;
        assertEquals(loop, loop2);
    }

    /** Tests the latex command. */
    public void testLatexCommand() {
        Dimension dim = new Dimension(112, 112);

        String expected = "\\Arc(100,12)(50.931,133,493)";
        String actual = loop.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

        loop.setPaintArrow(true);
        expected = "\\Arc[arrow,arrowpos=0,arrowlength=5,arrowwidth=2,arrowinset=0.2](100,12)(50.931,133,493)";
        actual = loop.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

        loop.setFlip(true);
        expected = "\\Arc[arrow,arrowpos=0,arrowlength=5,arrowwidth=2,arrowinset=0.2,flip](100,12)(50.931,133,493)";
        actual = loop.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

        // reset to initial state
        loop.setPaintArrow(false);
        loop.setFlip(false);

        loop.setDoubleLine(true);
        loop.setDLSeparation(4.f);

        expected = "\\Arc[double,sep=4](100,12)(50.931,133,493)";
        actual = loop.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

        loop.setRelWAndH(0, 0);
        expected = "%";
        actual = loop.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

    }

    /** Tests axodraw. */
    public void testAxodraw() {
        JaxoAxodraw axo = new JaxoAxodraw();
        JaxoObject ob = axo.getJaxoObject("\\SetWidth{1.0}");
        assertNull(ob);
        String command;

        command = "\\ArrowArc(100,200)(52.33,154,514)";
        loop = (JaxoFLoop) axo.getJaxoObject(command);
        checkLoopParameters(false);

        command = "\\ArrowArcn(100,200)(52.33,154,514)";
        loop = (JaxoFLoop) axo.getJaxoObject(command);
        checkLoopParameters(false);

        command = "\\CArc(100,200)(52.33,154,514)";
        loop = (JaxoFLoop) axo.getJaxoObject(command);
        checkLoopParameters(false);

        command =
            "\\ArrowArc(100,200)(54.33,154,514)\\ArrowArc(100,200)(50.33,154,514)%%JaxoDrawID:DoubleLine(3)";
        loop = (JaxoFLoop) axo.getJaxoObject(command);
        checkLoopParameters(true);

        command =
            "\\ArrowArcn(100,200)(60.69,517,157)\\ArrowArcn(100,200)(56.69,517,157)%%JaxoDrawID:DoubleLine(3)";
        loop = (JaxoFLoop) axo.getJaxoObject(command);
        checkLoopParameters(true);

        command =
            "\\CArc(100,200)(50.84,133,493)\\CArc(100,200)(46.84,133,493)%%JaxoDrawID:DoubleLine(3)";
        loop = (JaxoFLoop) axo.getJaxoObject(command);
        checkLoopParameters(true);

    }

    private void checkLoopParameters(boolean doubleLine) {
        assertEquals("FLoop doesn't have the right color!", JaxoColor.BLACK,
            loop.getColor());
        assertTrue("FLoop doesn't have the right stroke!",
            Math.abs(loop.getStrokeWidth() - 1.f) < 0.0001f);
        assertEquals("FLoop doesn't have the right x!", 100, loop.getX());
        assertEquals("FLoop doesn't have the right y!", -200, loop.getY());
        assertEquals("Mismatch in FLoop doubleLine!", doubleLine,
            loop.isDoubleLine());

        float dlsep = 2.f;
        if (doubleLine) {
            dlsep = 3.f;
        }

        assertTrue("Mismatch in FLoop doubleLine separation!",
            Math.abs(loop.getDLSeparation() - dlsep) < 0.0001f);

    }

    /** Tests the getBounds() method. */
    public void testGetBounds() {
        Rectangle bBox = loop.getBounds();
        Rectangle expected = new Rectangle(48, 48, 104, 104);
        assertEquals("Bounding box not correct!", expected, bBox);
        loop.setPaintArrow(true);
        bBox = loop.getBounds();
        expected.setRect(48, 48, 104, 104);
        assertEquals("Arc Bounding box not correct!", expected, bBox);
        loop.setDoubleLine(true);
        bBox = loop.getBounds();
        expected.setRect(47, 47, 106, 106);
        assertEquals("DL Bounding box not correct!", expected, bBox);

        assertNotNull(loop.getBounds2D());
    }

    /** Tests createNewObject in JaxoObjectFactory. */
    public void testCreateNewObject() {
        Point p = new Point(50, 50);
        JaxoObject ob =
            JaxoObjectFactory.newObject(JaxoConstants.FLOOP, new Point[] {p, p});

        assertTrue("createNewObject did not return a FLoop!",
            (ob instanceof JaxoFLoop));

        loop = (JaxoFLoop) ob;
        assertEquals("FLoop doesn't have the right x!", 50, loop.getX());
        assertEquals("FLoop doesn't have the right y!", 50, loop.getY());
        assertEquals("FLoop doesn't have the right color!",
            JaxoColor.getColor(JaxoPrefs.getStringPref(JaxoPrefs.PREF_COLOR)),
            loop.getColor());
        assertTrue("FLoop doesn't have the right stroke!",
            loop.strokeIs(JaxoPrefs.getFloatPref(JaxoPrefs.PREF_LINEWIDTH)));
        assertEquals("FLoop doesn't have the right arrow!",
            JaxoPrefs.getBooleanPref(JaxoPrefs.PREF_ARROW), loop.isPaintArrow());
        assertEquals("FLoop doesn't have the right flip!", false,
            loop.isFlip());
        assertEquals("FLoop doesn't have the right double line!", false,
            loop.isDoubleLine());
        assertTrue("FLoop doesn't have the right dlsep!",
            loop.dlSepIs(JaxoPrefs.getFloatPref(JaxoPrefs.PREF_DLSEP)));

    }

    /** {@inheritDoc} */
    public String className() {
        return "JaxoFLoop";
    }

    /** {@inheritDoc} */
    public void testJaxo20Compat() {
        JaxoFLoop object = (JaxoFLoop) instanceFrom20OpenGraph(JaxoFLoop.class);
        assertNotNull(object);
        check20Object("opened wrong object!", object);

        object = (JaxoFLoop) instanceFrom20ImportGraph(JaxoFLoop.class);
        assertNotNull(object);
        check20Object("imported wrong object!", object);
    }

    private void check20Object(String message, JaxoFLoop object) {

        assertEquals(message, 231, object.getX());
        assertEquals(message, 63, object.getY());
        assertEquals(message, 29, (int) Math.round(object.getRadius()));

        assertTrue(message, object.isDoubleLine());
        assertTrue(message, object.dlSepIs(3.f));
        assertTrue(message, object.strokeIs(1.4f));

        assertEquals(message, JaxoColor.PLUM, object.getColor());

        assertTrue(message, object.isPaintArrow());
        assertTrue(message, object.isFlip());
        assertArrow(message, object.getArrow(), 15.f, 6.f, 0.2f);
    }
}
