/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.object.loop;

import java.awt.Dimension;
import java.awt.Point;
import java.awt.Rectangle;

import net.sf.jaxodraw.JaxoTestUtils;
import net.sf.jaxodraw.io.imports.JaxoAxodraw;
import net.sf.jaxodraw.object.JaxoObject;
import net.sf.jaxodraw.object.JaxoObjectFactory;
import net.sf.jaxodraw.object.arc.JaxoFArc;
import net.sf.jaxodraw.util.JaxoColor;
import net.sf.jaxodraw.util.JaxoConstants;
import net.sf.jaxodraw.util.JaxoPrefs;


/** Tests JaxoGLoop. */
public class JaxoGLoopTest extends AbstractJaxoLoopObjectTest {
    private JaxoGLoop loop = new JaxoGLoop();

    /** Constructor: just calls super().
     * @param testName The name of this test.
     */
    public JaxoGLoopTest(String testName) {
        super(testName);
    }

    /** JUnit setup: initialize the object. */
    @Override
    protected void setUp() {
        super.setUp();
        init();
    }

    private void init() {
        loop.setLocation(58, 71);
        loop.setRelWAndH(-42, -29);
        loop.setColor(JaxoColor.RED);
        loop.setStrokeWidth(1.f);
        loop.setDash(2.f);
        loop.setPaintArrow(false);
        loop.setFlip(false);
        loop.setDoubleLine(false);
        loop.setDLSeparation(2.f);
    }

    /** {@inheritDoc} */
    public JaxoObject getInstance() {
        return loop;
    }

    /** Tests serialization. */
    public void testSerialization() {
        loop.setStrokeWidth(1.6f);
        loop.setDash(3.f);
        loop.setPaintArrow(true);
        loop.setFlip(true);
        loop.setDoubleLine(true);
        JaxoTestUtils.checkSerializedEquality(loop);
    }

    /** Tests that a copy of an object is still the same object. */
    public void testCopy() {
        JaxoGLoop loop2 = new JaxoGLoop();
        loop2.setState(loop);

        loop.setStrokeWidth(1.6f);
        loop.setDash(3.f);
        loop.setPaintArrow(true);
        loop.setFlip(true);
        loop.setDoubleLine(true);
        JaxoTestUtils.checkCopyEquality(loop);

        assertFalse(loop2.isCopy(loop));

        loop2.setStrokeWidth(1.6f);
        loop2.setDash(3.f);
        loop2.setPaintArrow(true);
        loop2.setFlip(true);
        loop2.setDoubleLine(true);
        assertTrue(loop2.isCopy(loop));

        JaxoFArc arc = new JaxoFArc();
        assertFalse(loop.isCopy(arc));

    }

    /** Tests identity. */
    public void testIdentity() {
        JaxoGLoop loop2 = loop;
        assertEquals(loop, loop2);
    }

    /** Tests the latex command. */
    public void testLatexCommand() {
        Dimension dim = new Dimension(102, 102);

        String expected = "\\Arc[dash,dashsize=2](58,31)(51.039,325,685)";
        String actual = loop.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

        loop.setPaintArrow(true);
        expected = "\\Arc[dash,dashsize=2,arrow,arrowpos=0,arrowlength=5,arrowwidth=2,arrowinset=0.2](58,31)(51.039,325,685)";
        actual = loop.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

        loop.setFlip(true);
        expected = "\\Arc[dash,dashsize=2,arrow,arrowpos=0,arrowlength=5,arrowwidth=2,arrowinset=0.2,flip](58,31)(51.039,325,685)";
        actual = loop.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

        // reset to initial state
        loop.setPaintArrow(false);
        loop.setFlip(false);

        loop.setDoubleLine(true);
        loop.setDLSeparation(4.f);

        expected = "\\Arc[dash,dashsize=2,double,sep=4](58,31)(51.039,325,685)";
        actual = loop.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

        loop.setRelWAndH(0, 0);
        expected = "%";
        actual = loop.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

    }

    /** Tests axodraw. */
    public void testAxodraw() {
        JaxoAxodraw axo = new JaxoAxodraw();
        JaxoObject ob = axo.getJaxoObject("\\SetWidth{1.0}");
        assertNull(ob);
        axo.checkRequired("\\begin{picture}(102,102) (48,-48)");
        String command;

        command = "\\DashCArc(100,2)(51.04,325,685){2}";
        loop = (JaxoGLoop) axo.getJaxoObject(command);
        checkLoopParameters(false);

        command = "\\DashArrowArc(100,2)(51.04,325,685){2}";
        loop = (JaxoGLoop) axo.getJaxoObject(command);
        checkLoopParameters(false);

        command = "\\DashArrowArcn(100,2)(51.04,685,325){2}";
        loop = (JaxoGLoop) axo.getJaxoObject(command);
        checkLoopParameters(false);

        axo.checkRequired("\\begin{picture}(108,108) (45,-45)");

        command =
            "\\DashCArc(100,8)(54.04,325,685){2.0}\\DashCArc(100,8)(48.04,325,685){2.0}%%JaxoDrawID:DoubleLine(3)";
        loop = (JaxoGLoop) axo.getJaxoObject(command);
        checkLoopParameters(true);

        command =
            "\\DashArrowArc(100,8)(54.04,325,685){2.0}\\DashArrowArc(100,8)(48.04,325,685){2.0}%%JaxoDrawID:DoubleLine(3)";
        loop = (JaxoGLoop) axo.getJaxoObject(command);
        checkLoopParameters(true);

        command =
            "\\DashArrowArcn(100,8)(54.04,685,325){2.0}\\DashArrowArcn(100,8)(48.04,685,325){2.0}%%JaxoDrawID:DoubleLine(3)";
        loop = (JaxoGLoop) axo.getJaxoObject(command);
        checkLoopParameters(true);

    }

    private void checkLoopParameters(boolean doubleLine) {
        assertEquals("GLoop doesn't have the right color!", JaxoColor.BLACK,
            loop.getColor());
        assertTrue("GLoop doesn't have the right stroke!", loop.strokeIs(1.f));
        assertTrue("GLoop doesn't have the right dash!", loop.dashIs(2.f));
        assertEquals("GLoop doesn't have the right x!", 100, loop.getX());
        assertEquals("GLoop doesn't have the right y!", 100, loop.getY());
        assertEquals("Mismatch in GLoop doubleLine!", doubleLine,
            loop.isDoubleLine());

        float dlsep = 2.f;
        if (doubleLine) {
            dlsep = 3.f;
        }

        assertTrue("Mismatch in GLoop doubleLine separation!",
            loop.dlSepIs(dlsep));

    }

    /** Tests the getBounds() method. */
    public void testGetBounds() {
        Rectangle bBox = loop.getBounds();
        Rectangle expected = new Rectangle(6, 19, 104, 104);
        assertEquals("Bounding box not correct!", expected, bBox);

        loop.setPaintArrow(false);
        bBox = loop.getBounds();
        expected.setRect(6, 19, 104, 104);
        assertEquals("Arc Bounding box not correct!", expected, bBox);

        loop.setDoubleLine(true);
        bBox = loop.getBounds();
        expected.setRect(5, 18, 106, 106);
        assertEquals("DL Bounding box not correct!", expected, bBox);

        assertNotNull(loop.getBounds2D());
    }

    /** Tests createNewObject in JaxoObjectFactory. */
    public void testCreateNewObject() {
        Point p = new Point(50, 50);
        JaxoObject ob =
            JaxoObjectFactory.newObject(JaxoConstants.GLOOP, new Point[] {p, p});

        assertTrue("createNewObject did not return a GLoop!",
            (ob instanceof JaxoGLoop));

        loop = (JaxoGLoop) ob;
        assertEquals("GLoop doesn't have the right x!", 50, loop.getX());
        assertEquals("GLoop doesn't have the right y!", 50, loop.getY());
        assertEquals("GLoop doesn't have the right color!",
            JaxoColor.getColor(JaxoPrefs.getStringPref(JaxoPrefs.PREF_COLOR)),
            loop.getColor());
        assertTrue("GLoop doesn't have the right stroke!",
            loop.strokeIs(JaxoPrefs.getFloatPref(JaxoPrefs.PREF_LINEWIDTH)));
        assertEquals("GLoop doesn't have the right arrow!",
            JaxoPrefs.getBooleanPref(JaxoPrefs.PREF_ARROW), loop.isPaintArrow());
        assertEquals("GLoop doesn't have the right flip!", false,
            loop.isFlip());
        assertEquals("GLoop doesn't have the right double line!", false,
            loop.isDoubleLine());
        assertTrue("GLoop doesn't have the right dlsep!",
            loop.dlSepIs(JaxoPrefs.getFloatPref(JaxoPrefs.PREF_DLSEP)));
        assertTrue("GLoop doesn't have the right dash!", loop.dashIs(2.f));

    }

    /** {@inheritDoc} */
    public String className() {
        return "JaxoGLoop";
    }

    /** {@inheritDoc} */
    public void testJaxo20Compat() {
        JaxoGLoop object = (JaxoGLoop) instanceFrom20OpenGraph(JaxoGLoop.class);
        assertNotNull(object);
        check20Object("opened wrong object!", object);

        object = (JaxoGLoop) instanceFrom20ImportGraph(JaxoGLoop.class);
        assertNotNull(object);
        check20Object("imported wrong object!", object);
    }

    private void check20Object(String message, JaxoGLoop object) {
        assertEquals(message, 209, object.getX());
        assertEquals(message, 223, object.getY());
        assertEquals(message, 40, (int) Math.round(object.getRadius()));

        assertFalse(message, object.isDoubleLine());
        assertTrue(message, object.strokeIs(2.f));
        assertEquals(message, 1.997f, object.getDash());

        assertEquals(message, JaxoColor.CORNFLOWER_BLUE, object.getColor());

        assertTrue(message, object.isPaintArrow());
        assertFalse(message, object.isFlip());
        assertArrow(message, object.getArrow(), 5.f, 2.f, 0.2f);
    }
}
