/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.object.loop;

import java.awt.Dimension;
import java.awt.Point;
import java.awt.Rectangle;

import net.sf.jaxodraw.JaxoTestUtils;
import net.sf.jaxodraw.io.imports.JaxoAxodraw;
import net.sf.jaxodraw.object.JaxoObject;
import net.sf.jaxodraw.object.JaxoObjectFactory;
import net.sf.jaxodraw.object.arc.JaxoFArc;
import net.sf.jaxodraw.util.JaxoColor;
import net.sf.jaxodraw.util.JaxoConstants;
import net.sf.jaxodraw.util.JaxoPrefs;


/** Tests JaxoPLoop. */
public class JaxoPLoopTest extends AbstractJaxoLoopObjectTest {
    private JaxoPLoop loop = new JaxoPLoop();

    /** Constructor: just calls super().
     * @param testName The name of this test.
     */
    public JaxoPLoopTest(String testName) {
        super(testName);
    }

    /** JUnit setup: initialize the object. */
    @Override
    protected void setUp() {
        super.setUp();
        init();
    }

    private void init() {
        loop.setLocation(100, 69);
        loop.setRelWAndH(40, -31);
        loop.setColor(JaxoColor.RED);
        loop.setStrokeWidth(1.f);
        loop.setAmp(15);
        loop.setWiggles(36);
        loop.setDoubleLine(false);
        loop.setDLSeparation(2.f);
    }

    /** {@inheritDoc} */
    public JaxoObject getInstance() {
        return loop;
    }

    /** Tests serialization. */
    public void testSerialization() {
        loop.setStrokeWidth(1.6f);
        loop.setAmp(25);
        loop.setDoubleLine(true);
        JaxoTestUtils.checkSerializedEquality(loop);
    }

    /** Tests that a copy of an object is still the same object. */
    public void testCopy() {
        JaxoPLoop loop2 = new JaxoPLoop();
        loop2.setState(loop);

        loop.setStrokeWidth(1.6f);
        loop.setAmp(25);
        loop.setDoubleLine(true);
        JaxoTestUtils.checkCopyEquality(loop);

        assertFalse(loop2.isCopy(loop));

        loop2.setStrokeWidth(1.6f);
        loop2.setAmp(25);
        loop2.setDoubleLine(true);
        assertTrue(loop2.isCopy(loop));

        JaxoFArc arc = new JaxoFArc();
        assertFalse(loop.isCopy(arc));

    }

    /** Tests identity. */
    public void testIdentity() {
        JaxoPLoop loop2 = loop;
        assertEquals(loop, loop2);
    }

    /** Tests the latex command. */
    public void testLatexCommand() {
        Dimension dim = new Dimension(117, 116);

        String expected = "\\PhotonArc(100,47)(50.606,38,398){7.5}{36}";
        String actual = loop.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

        // reset to initial state
        loop.setPaintArrow(false);
        loop.setFlip(false);

        loop.setDoubleLine(true);
        loop.setDLSeparation(4.f);

        expected = "\\PhotonArc[double,sep=4](100,47)(50.606,38,398){7.5}{36}";
        actual = loop.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

        loop.setRelWAndH(0, 0);
        expected = "%";
        actual = loop.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

    }

    /** Tests axodraw. */
    public void testAxodraw() {
        JaxoAxodraw axo = new JaxoAxodraw();
        JaxoObject ob = axo.getJaxoObject("\\SetWidth{1.0}");
        assertNull(ob);
        axo.checkRequired("\\begin{picture}(116,116) (41,-41)");
        String command;

        command = "\\PhotonArc(100,16)(50.61,38,398){7.5}{36}";
        loop = (JaxoPLoop) axo.getJaxoObject(command);
        checkLoopParameters(false);

        axo.checkRequired("\\begin{picture}(121,121) (39,-39)");

        command =
            "\\PhotonArc(100,21)(52.61,38,398){7.5}{36}\\PhotonArc(100,21)(48.61,38,398){7.5}{36}%%JaxoDrawID:DoubleLine(3)";
        loop = (JaxoPLoop) axo.getJaxoObject(command);
        checkLoopParameters(true);

    }

    private void checkLoopParameters(boolean doubleLine) {
        assertEquals("PLoop doesn't have the right color!", JaxoColor.BLACK,
            loop.getColor());
        assertEquals("PLoop doesn't have the right amp!", 15, loop.getAmp());
        assertEquals("PLoop doesn't have the right wiggles!", 36, loop.getWiggles());
        assertEquals("PLoop doesn't have the right x!", 100, loop.getX());
        assertEquals("PLoop doesn't have the right y!", 100, loop.getY());
        assertEquals("Mismatch in PLoop doubleLine!", doubleLine,
            loop.isDoubleLine());

        float dlsep = 2.f;

        if (doubleLine) {
            dlsep = 3.f;
        }

        assertTrue("Mismatch in PLoop doubleLine separation!",
            loop.dlSepIs(dlsep));
    }

    /** Tests the getBounds() method. */
    public void testGetBounds() {
        Rectangle bBox = loop.getBounds();
        Rectangle expected = new Rectangle(41, 10, 118, 118);
        assertEquals("Bounding box not correct!", expected, bBox);
        loop.setDoubleLine(true);
        bBox = loop.getBounds();
        expected.setRect(40, 9, 120, 120);
        assertEquals("DL Bounding box not correct!", expected, bBox);

        assertNotNull(loop.getBounds2D());
    }

    /** Tests createNewObject in JaxoObjectFactory. */
    public void testCreateNewObject() {
        Point p = new Point(50, 50);
        JaxoObject ob =
            JaxoObjectFactory.newObject(JaxoConstants.PLOOP, new Point[] {p, p});

        assertTrue("createNewObject did not return a PLoop!",
            (ob instanceof JaxoPLoop));

        loop = (JaxoPLoop) ob;
        assertEquals("PLoop doesn't have the right x!", 50, loop.getX());
        assertEquals("PLoop doesn't have the right y!", 50, loop.getY());
        assertEquals("PLoop doesn't have the right color!",
            JaxoColor.getColor(JaxoPrefs.getStringPref(JaxoPrefs.PREF_COLOR)),
            loop.getColor());
        assertTrue("PLoop doesn't have the right stroke!",
            loop.strokeIs(JaxoPrefs.getFloatPref(JaxoPrefs.PREF_LINEWIDTH)));
        assertEquals("PLoop doesn't have the right double line!", false,
            loop.isDoubleLine());
        assertTrue("PLoop doesn't have the right dlsep!",
            loop.dlSepIs(JaxoPrefs.getFloatPref(JaxoPrefs.PREF_DLSEP)));
        assertEquals("PLoop doesn't have the right amplitude!",
            JaxoPrefs.getIntPref(JaxoPrefs.PREF_AMPLITUDE), loop.getAmp());
    }

    /** {@inheritDoc} */
    public String className() {
        return "JaxoPLoop";
    }

    /** {@inheritDoc} */
    public void testJaxo20Compat() {
        JaxoPLoop object = (JaxoPLoop) instanceFrom20OpenGraph(JaxoPLoop.class);
        assertNotNull(object);
        check20Object("opened wrong object!", object);

        object = (JaxoPLoop) instanceFrom20ImportGraph(JaxoPLoop.class);
        assertNotNull(object);
        check20Object("imported wrong object!", object);
    }

    private void check20Object(String message, JaxoPLoop object) {
        assertEquals(message, 216, object.getX());
        assertEquals(message, 350, object.getY());
        assertEquals(message, 31, (int) Math.round(object.getRadius()));

        assertTrue(message, object.isDoubleLine());
        assertTrue(message, object.dlSepIs(3.f));
        assertTrue(message, object.strokeIs(1.4f));

        assertEquals(message, JaxoColor.PERIWINKLE, object.getColor());

        assertEquals(message, 13, object.getAmp());
        assertEquals(message, 12, object.getWiggles());
    }
}
