/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.object.text;


import net.sf.jaxodraw.object.AbstractJaxoObjectTest;
import net.sf.jaxodraw.util.JaxoConstants;


/**
 * Base Test for JaxoObjects.
 */
public abstract class AbstractJaxoTextObjectTest extends AbstractJaxoObjectTest {
    private JaxoTextObject instance;

    /**
     * Constructor.
     *
     * @param testName The name of this test.
     */
    protected AbstractJaxoTextObjectTest(String testName) {
        super(testName);
    }

    /** JUnit setup: get a new instance of the object. */
    @Override
    protected void setUp() {
        super.setUp();
        this.instance = (JaxoTextObject) getInstance();
    }

      //
     // tests
    //

    /** Tests moveBy. */
    public void testMoveBy() {
        instance.setLocation(100, 100);
        instance.moveBy(23, -45);
        assertEquals("Wrong moveBy x!", 123, instance.getX());
        assertEquals("Wrong moveBy y!", 55, instance.getY());

        instance.moveBy(0, 0);
        assertEquals("Wrong moveBy x!", 123, instance.getX());
        assertEquals("Wrong moveBy y!", 55, instance.getY());
    }

    /** Tests getGrabbedHandle. */
    public void testGetGrabbedHandle() {
        instance.setLocation(100, 100);
        checkHandle(100, 100, JaxoTextObject.SELECT_P1);
        checkHandle(200, 100, JaxoTextObject.SELECT_NONE);
    }

    /** Tests smallestDistanceTo. */
    public void testSmallestDistanceTo() {
        instance.setLocation(100, 100);
        float dist = instance.smallestDistanceTo(103, 104);
        assertEquals("Wrong distance", dist, 5.f);
        dist = instance.smallestDistanceTo(150, 100);
        assertEquals("Wrong distance", dist, 50.f);
    }

    /** Tests latexWidth. */
    public void testLatexWidth() {
        String expected = "%";
        String actual = instance.latexWidth();
        assertEquals("Latex SetWidth commands do not match!", expected, actual);
    }

    /** Tests rescaleObject. */
    public void testRescaleObject() {
        instance.setLocation(100, 100);
        instance.rescaleObject(instance.getX(), instance.getY(), 2.f);
        assertEquals("Text x didn't get rescaled right!", 100, instance.getX());
        assertEquals("Text y didn't get rescaled right!", 100, instance.getY());
    }

    /** Tests canBeSelected. */
    public void testCanBeSelected() {
        int mode = JaxoConstants.MOVE;
        int handle = JaxoTextObject.SELECT_P1;
        assertTrue("P1 in MOVE mode", instance.canBeSelected(handle, mode));

        mode = JaxoConstants.RESIZE;
        assertFalse("P1 in RESIZE mode", instance.canBeSelected(handle, mode));
    }

    /** Tests all getters/setters. */
    public void testGetSetXY() {
        assertEquals("Wrong PointCount!", 1, instance.getPointCount());
        try {
            instance.setPointCount(2);
            fail("setPointCount(2) should throw IllegalArgumentException");
        } catch (IllegalArgumentException e) {
            // expected, now check it doesn't throw with proper value
            instance.setPointCount(1);
        }
    }

      //
     // abstract
    //

    /** Tests serialization. */
    public abstract void testSerialization();

}
