/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.object.text;


import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Point;

import net.sf.jaxodraw.JaxoTestUtils;
import net.sf.jaxodraw.object.JaxoObject;
import net.sf.jaxodraw.object.JaxoObjectFactory;
import net.sf.jaxodraw.object.loop.JaxoFLoop;
import net.sf.jaxodraw.util.JaxoColor;
import net.sf.jaxodraw.util.JaxoConstants;
import net.sf.jaxodraw.util.JaxoPrefs;


/** Tests JaxoPSText. */
public class JaxoPSTextTest extends AbstractJaxoTextObjectTest {
    private JaxoPSText text = new JaxoPSText();
    private final Font font = new Font("Default", 2, 18);

    /** Constructor: just calls super().
     * @param testName The name of this test.
     */
    public JaxoPSTextTest(String testName) {
        super(testName);
    }

    /** JUnit setup: initialize the object. */
    @Override
    protected void setUp() {
        super.setUp();
        init();
    }

    private void init() {
        text.setLocation(100, 100);
        text.setColor(JaxoColor.BLACK);
        text.setRotationAngle(0);
        text.setTextString("A_{\\mu}^{\\nu}");
        text.setFont(font);
    }

    /** {@inheritDoc} */
    public JaxoObject getInstance() {
        return text;
    }

    /** Tests serialization. */
    public void testSerialization() {
        text.setRotationAngle(15);
        text.setTextString("Feynman");
        JaxoTestUtils.checkSerializedEquality(text);
    }

    /** Tests that a copy of an object is still the same object. */
    public void testCopy() {
        JaxoPSText text2 = new JaxoPSText();
        text2.setState(text);

        text.setRotationAngle(15);
        text.setTextString("brackets: {,},\\{,\\}");
        JaxoTestUtils.checkCopyEquality(text);

        assertFalse(text2.isCopy(text));

        text2.setRotationAngle(15);
        text2.setTextString("brackets: {,},\\{,\\}");
        assertTrue(text2.isCopy(text));

        text2.setLocation(2, 2);
        assertFalse(text2.isCopy(text));

        JaxoFLoop loop = new JaxoFLoop();
        assertFalse(text.isCopy(loop));

    }

    /** Tests identity. */
    public void testIdentity() {
        JaxoPSText text2 = text;
        assertEquals(text, text2);
    }

    /** Tests the latex command. */
    public void testLatexCommand() {
        Dimension dim = new Dimension(100, 100);

        String expected = "% There is a postscript text here!";
        String actual = text.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

    }

    /** Tests createNewObject in JaxoObjectFactory. */
    public void testCreateNewObject() {
        Point p = new Point(50, 50);
        JaxoObject ob =
            JaxoObjectFactory.newObject(JaxoConstants.TEXT, new Point[] {p});

        assertTrue("createNewObject did not return a PSText!",
            (ob instanceof JaxoPSText));

        text = (JaxoPSText) ob;
        assertEquals("PSText doesn't have the right x!", 50, text.getX());
        assertEquals("PSText doesn't have the right y!", 50, text.getY());
        assertEquals("PSText doesn't have the right color!",
            JaxoColor.getColor(JaxoPrefs.getStringPref(JaxoPrefs.PREF_COLOR)),
            text.getColor());
        assertEquals("PSText doesn't have the right text!",
            "", text.getTextString());
    }

    /** {@inheritDoc} */
    public void testAxodraw() {
        // nothing
    }

    /** {@inheritDoc} */
    public void testGetBounds() {
        // this works on Linux but fails on Windows, tell me why...
        //assertEquals(32, text.getBounds().width);
        //assertEquals(28, text.getBounds().height);

        assertNotNull(text.getBounds2D());

        assertEquals(0, text.getWidth());
        assertEquals(0, text.getHeight());

        assertNotNull(text.getBoundsWithHandles(defaultHandle()));
    }

    /** {@inheritDoc} */
    public String className() {
        return "JaxoPSText";
    }

    /**
     * Tests various setParameter methods.
     *
     * This actually belongs into JaxoObject but it's safer to do it with
     * a concrete class to avoid surprises with overriding methods
     * (eg setColor() gets overridden by FillObjects).
     */
    public void testSetParameter() {
        getInstance().setParameter("color", Color.class, Color.ORANGE);
        assertEquals(Color.ORANGE, getInstance().getColor());
        getInstance().setParameter("color", Color.PINK);
        assertEquals(Color.PINK, getInstance().getColor());

        getInstance().setParameter("x", 17);
        assertEquals(17, getInstance().getX());
        getInstance().setParameter("asMarked", true);
        assertTrue(getInstance().isMarked());


        getInstance().setParameter("textString", "testText");
        assertEquals("testText", text.getTextString());

        try {
            getInstance().setParameter("pumuckl", String.class, null, true);
            fail("unknown parameter set");
        } catch (UnsupportedOperationException e) {
            assertNotNull(e);
            getInstance().setParameter("pumuckl", String.class, null, false);
        }

        try {
            getInstance().setParameter("", "testText");
            fail("Empty parameter name should throw!");
        } catch (IllegalArgumentException e) {
            assertNotNull(e);
        }
    }

    /** {@inheritDoc} */
    public void testJaxo20Compat() {
        JaxoPSText object =
                (JaxoPSText) instanceFrom20OpenGraph(JaxoPSText.class);
        assertNotNull(object);

        assertEquals(474, object.getX());
        assertEquals(409, object.getY());

        assertEquals("Hello", object.getTextString());
        String family = object.getFont().getFamily();
        // java 4 returns "Default", java 5 "Dialog"...
        if (!("Dialog".equals(family) || "Default".equals(family))) {
            fail();
        }
        assertEquals(Font.BOLD + Font.ITALIC, object.getFont().getStyle());
        assertEquals(22, object.getFont().getSize());

        assertEquals(8, object.getRotationAngle());

        assertEquals(JaxoColor.RED, object.getColor());

        object = (JaxoPSText) instanceFrom20ImportGraph(JaxoPSText.class);
        assertNull(object);
    }

    /** {@inheritDoc} */
    public String[] serializableProperties() {
        return new String[] {
            "font", "color", "pointCount", "points",
            "rotationAngle", "textString"
        }; // 6
    }
}
