/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.object.vertex;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Point;
import java.awt.Rectangle;

import net.sf.jaxodraw.JaxoTestUtils;
import net.sf.jaxodraw.io.imports.JaxoAxodraw;
import net.sf.jaxodraw.object.JaxoObject;
import net.sf.jaxodraw.object.JaxoObjectFactory;
import net.sf.jaxodraw.object.loop.JaxoFLoop;
import net.sf.jaxodraw.util.JaxoColor;
import net.sf.jaxodraw.util.JaxoConstants;
import net.sf.jaxodraw.util.JaxoPrefs;


/** Tests JaxoVertexT2. */
public class JaxoVertexT2Test extends AbstractJaxoVertexTest {
    private JaxoVertexT2 vertex = new JaxoVertexT2();

    /** Constructor: just calls super().
     * @param testName The name of this test.
     */
    public JaxoVertexT2Test(String testName) {
        super(testName);
    }

    /** JUnit setup: initialize the object. */
    @Override
    protected void setUp() {
        super.setUp();
        init();
    }

    private void init() {
        vertex.setLocation(100, 100);
        vertex.setRelWAndH(100, 100);
        vertex.setStrokeWidth(1.f);
        vertex.setColor(JaxoColor.GREEN);
        vertex.setFillColor(JaxoColor.RED);
    }

    /** {@inheritDoc} */
    public JaxoObject getInstance() {
        return vertex;
    }

    /** Tests serialization. */
    public void testSerialization() {
        vertex.setStrokeWidth(1.5f);
        vertex.setColor(JaxoColor.RED);
        vertex.setFillColor(JaxoColor.GREEN);
        JaxoTestUtils.checkSerializedEquality(vertex);
    }

    /** Tests that a copy of an object is still the same object. */
    public void testCopy() {
        JaxoVertexT2 vertex2 = new JaxoVertexT2();
        vertex2.setState(vertex);

        vertex.setStrokeWidth(1.5f);
        vertex.setColor(JaxoColor.RED);
        vertex.setFillColor(JaxoColor.GREEN);
        JaxoTestUtils.checkCopyEquality(vertex);

        assertFalse(vertex2.isCopy(vertex));

        vertex2.setStrokeWidth(1.5f);
        vertex2.setColor(JaxoColor.RED);
        vertex2.setFillColor(JaxoColor.GREEN);
        assertTrue(vertex2.isCopy(vertex));

        JaxoFLoop loop = new JaxoFLoop();
        assertFalse(vertex.isCopy(loop));

    }

    /** Tests identity. */
    public void testIdentity() {
        JaxoVertexT2 vertex2 = vertex;
        assertEquals(vertex, vertex2);
    }

    /** Tests the latex command. */
    public void testLatexCommand() {
        Dimension dim = new Dimension(100, 100);

        vertex.setRelWAndH(50, 0);
        vertex.setFillColor(JaxoColor.FOREST_GREEN);
        vertex.setColor(JaxoColor.CERULEAN);
        String expected =
            "\\COval(100,0)(50,50)(0.0){Cerulean}{ForestGreen}\\Line(100,-35.355)(100,35.355)\\Line(64.645,0)(135.355,0)";
        String actual = vertex.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

        vertex.setFillColor(JaxoColor.GRAYSCALE180);
        expected =
            "\\GOval(100,0)(50,50)(0.0){0.705}\\Line(100,-35.355)(100,35.355)\\Line(64.645,0)(135.355,0)";
        actual = vertex.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

        vertex.setFilled(false);
        expected = "\\Oval(100,0)(50,50)(0.0)\\Line(100,-35.355)(100,35.355)\\Line(64.645,0)(135.355,0)";
        actual = vertex.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

        expected = "%";
        actual = vertex.latexCommand(100.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

    }

    /** Tests axodraw. */
    public void testAxodraw() {
        JaxoAxodraw axo = new JaxoAxodraw();
        JaxoObject ob = axo.getJaxoObject("\\SetWidth{1.0}");
        assertNull(ob);
        String command;

        command =
            "\\COval(100,0)(50,50)(0.0){Cerulean}{ForestGreen}\\Line(100,-35.36)(100,35.36)\\Line(64.64,0)(135.36,0)";
        vertex = (JaxoVertexT2) axo.getJaxoObject(command);
        checkVertexParameters(JaxoColor.CERULEAN, JaxoColor.FOREST_GREEN);

        command =
            "\\GOval(100,0)(50,50)(0.0){0.705}\\Line(100,-35.36)(100,35.36)\\Line(64.64,0)(135.36,0)";
        vertex = (JaxoVertexT2) axo.getJaxoObject(command);
        checkVertexParameters(JaxoColor.BLACK, JaxoColor.GRAYSCALE180);

        command =
            "\\Oval(100,0)(50,50)(0.0)\\Line(100,-35.36)(100,35.36)\\Line(64.64,0)(135.36,0)";
        vertex = (JaxoVertexT2) axo.getJaxoObject(command);
        assertFalse("Vertex should not be filled!", vertex.isFilled());

    }

    private void checkVertexParameters(Color color, Color fillcolor) {
        assertEquals("VertexT2 doesn't have the right color!", color,
            vertex.getColor());
        assertEquals("VertexT2 doesn't have the right fill color!", fillcolor,
            vertex.getFillColor());
        assertTrue("VertexT2 doesn't have the right stroke!",
            vertex.strokeIs(1.f));
        assertEquals("VertexT2 doesn't have the right relw!", 50,
            vertex.getRelw());
        assertEquals("VertexT2 doesn't have the right relh!", 0,
            vertex.getRelh());
        assertEquals("VertexT2 doesn't have the right x!", 100, vertex.getX());
        assertEquals("VertexT2 doesn't have the right y!", 0, vertex.getY());
    }

    /** Tests the getBounds() method. */
    public void testGetBounds() {
        Rectangle bBox = vertex.getBounds();
        Rectangle expected = new Rectangle(-42, -42, 284, 284);
        assertEquals("Bounding box not correct!", expected, bBox);
        vertex.setFilled(false);
        bBox = vertex.getBounds();
        assertEquals("Bounding box still not correct!", expected, bBox);

        assertNotNull(vertex.getBounds2D());
    }

    /** Tests createNewObject in JaxoObjectFactory. */
    public void testCreateNewObject() {
        Point p = new Point(50, 50);
        JaxoObject ob =
            JaxoObjectFactory.newObject(JaxoConstants.VERTEXT2, new Point[] {p, p});

        assertTrue("createNewObject did not return a VertexT2!",
            (ob instanceof JaxoVertexT2));

        vertex = (JaxoVertexT2) ob;
        assertEquals("VertexT2 doesn't have the right x!", 50, vertex.getX());
        assertEquals("VertexT2 doesn't have the right y!", 50, vertex.getY());
        assertEquals("VertexT2 doesn't have the right color!",
            JaxoColor.getColor(JaxoPrefs.getStringPref(JaxoPrefs.PREF_COLOR)),
            vertex.getColor());
        assertEquals("VertexT2 doesn't have the right fillcolor!",
            JaxoColor.WHITE, vertex.getFillColor());

    }

    /** {@inheritDoc} */
    public String className() {
        return "JaxoVertexT2";
    }

    /** {@inheritDoc} */
    public void testJaxo20Compat() {
        JaxoVertexT2 object =
                (JaxoVertexT2) instanceFrom20OpenGraph(JaxoVertexT2.class);
        assertNotNull(object);
        check20Object("opened wrong object!", object);

        object =
                (JaxoVertexT2) instanceFrom20ImportGraph(JaxoVertexT2.class);
        assertNotNull(object);
        check20Object("imported wrong object!", object);
    }

    private void check20Object(String message, JaxoVertexT2 object) {
        assertEquals(message, 537, object.getX());
        assertEquals(message, 183, object.getY());
        assertEquals(message, 11, (int) Math.round(object.getRadius()));

        assertEquals(message, 5, object.getRotationAngle());

        assertEquals(message, JaxoColor.BLACK, object.getColor());
        assertEquals(message, JaxoColor.WHITE, object.getFillColor());
        assertTrue(message, object.isFilled());
    }

    /** {@inheritDoc} */
    @Override
    public String[] serializableProperties() {
        return new String[] {
            "pointCount", "points", "color", "fillColor", "filled",
            "strokeWidth", "rotationAngle"
        }; // 7
    }
}
