/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.object.vertex;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Point;
import java.awt.Rectangle;

import net.sf.jaxodraw.JaxoTestUtils;
import net.sf.jaxodraw.io.imports.JaxoAxodraw;
import net.sf.jaxodraw.object.JaxoObject;
import net.sf.jaxodraw.object.JaxoObjectFactory;
import net.sf.jaxodraw.object.loop.JaxoFLoop;
import net.sf.jaxodraw.util.JaxoColor;
import net.sf.jaxodraw.util.JaxoConstants;
import net.sf.jaxodraw.util.JaxoPrefs;


/** Tests JaxoVertexT3. */
public class JaxoVertexT3Test extends AbstractJaxoVertexTest {
    private JaxoVertexT3 vertex = new JaxoVertexT3();

    /** Constructor: just calls super().
     * @param testName The name of this test.
     */
    public JaxoVertexT3Test(String testName) {
        super(testName);
    }

    /** JUnit setup: initialize the object. */
    @Override
    protected void setUp() {
        super.setUp();
        init();
    }

    private void init() {
        vertex.setLocation(100, 100);
        vertex.setRelWAndH(100, 100);
        vertex.setStrokeWidth(1.f);
        vertex.setColor(JaxoColor.GREEN);
    }

    /** {@inheritDoc} */
    public JaxoObject getInstance() {
        return vertex;
    }

    /** Tests serialization. */
    public void testSerialization() {
        vertex.setColor(JaxoColor.RED);
        vertex.setStrokeWidth(2.f);
        JaxoTestUtils.checkSerializedEquality(vertex);
    }

    /** Tests that a copy of an object is still the same object. */
    public void testCopy() {
        vertex.setColor(JaxoColor.RED);
        vertex.setStrokeWidth(2.f);
        JaxoTestUtils.checkCopyEquality(vertex);

        JaxoVertexT3 vertex2 = new JaxoVertexT3();
        vertex2.setLocation(100, 100);
        vertex2.setRelWAndH(100, 100);
        vertex2.setStrokeWidth(1.f);
        vertex2.setColor(JaxoColor.GREEN);
        assertFalse(vertex2.isCopy(vertex));

        vertex2.setColor(JaxoColor.RED);
        vertex2.setStrokeWidth(2.f);
        assertTrue(vertex2.isCopy(vertex));

        JaxoFLoop loop = new JaxoFLoop();
        assertFalse(vertex.isCopy(loop));

    }

    /** Tests identity. */
    public void testIdentity() {
        JaxoVertexT3 vertex2 = vertex;
        assertEquals(vertex, vertex2);
    }

    /** Tests the latex command. */
    public void testLatexCommand() {
        Dimension dim = new Dimension(100, 100);

        vertex.setRelWAndH(50, 0);
        vertex.setColor(JaxoColor.CERULEAN);
        String expected =
            "\\CBox(150,-50)(50,50){Cerulean}{Cerulean}%%JaxoDrawID:Vertex";
        String actual = vertex.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

        expected = "%";
        actual = vertex.latexCommand(100.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

    }

    /** Tests axodraw. */
    public void testAxodraw() {
        JaxoAxodraw axo = new JaxoAxodraw();
        String command;

        command =
            "\\CBox(150,-50)(50,50){Cerulean}{Cerulean}%%JaxoDrawID:Vertex";
        vertex = (JaxoVertexT3) axo.getJaxoObject(command);
        checkVertexParameters(JaxoColor.CERULEAN);

    }

    private void checkVertexParameters(Color color) {
        assertEquals("VertexT3 doesn't have the right color!", color,
            vertex.getColor());
        assertEquals("VertexT3 doesn't have the right relw!", 35,
            vertex.getRelw());
        assertEquals("VertexT3 doesn't have the right relh!", 35,
            vertex.getRelh());
        assertEquals("VertexT3 doesn't have the right x!", 100, vertex.getX());
        assertEquals("VertexT3 doesn't have the right y!", 0, vertex.getY());
    }

    /** Tests the getBounds() method. */
    public void testGetBounds() {
        Rectangle bBox = vertex.getBounds();
        Rectangle expected = new Rectangle(-42, -42, 284, 284);
        assertEquals("Bounding box not correct!", expected, bBox);

        assertNotNull(vertex.getBounds2D());
    }

    /** Tests createNewObject in JaxoObjectFactory. */
    public void testCreateNewObject() {
        Point p = new Point(50, 50);
        JaxoObject ob =
            JaxoObjectFactory.newObject(JaxoConstants.VERTEXT3, new Point[] {p, p});

        assertTrue("createNewObject did not return a VertexT3!",
            (ob instanceof JaxoVertexT3));

        vertex = (JaxoVertexT3) ob;
        assertEquals("VertexT3 doesn't have the right x!", 50, vertex.getX());
        assertEquals("VertexT3 doesn't have the right y!", 50, vertex.getY());
        assertEquals("VertexT3 doesn't have the right color!",
            JaxoColor.getColor(JaxoPrefs.getStringPref(JaxoPrefs.PREF_COLOR)),
            vertex.getColor());
    }

    /** {@inheritDoc} */
    public String className() {
        return "JaxoVertexT3";
    }

    /** {@inheritDoc} */
    public void testJaxo20Compat() {
        JaxoVertexT3 object =
                (JaxoVertexT3) instanceFrom20OpenGraph(JaxoVertexT3.class);
        assertNotNull(object);
        check20Object("opened wrong object!", object);

        object =
                (JaxoVertexT3) instanceFrom20ImportGraph(JaxoVertexT3.class);
        assertNotNull(object);
        check20Object("imported wrong object!", object);
    }

    private void check20Object(String message, JaxoVertexT3 object) {
        assertEquals(message, 459, object.getX());
        assertEquals(message, 298, object.getY());
        assertEquals(message, 7, (int) Math.round(object.getRadius()));

        assertEquals(message, JaxoColor.BLUE, object.getColor());
    }
}
