/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.object.vertex;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Point;
import java.awt.Rectangle;

import net.sf.jaxodraw.JaxoTestUtils;
import net.sf.jaxodraw.io.imports.JaxoAxodraw;
import net.sf.jaxodraw.object.AbstractJaxo3PointObjectTest;
import net.sf.jaxodraw.object.JaxoObject;
import net.sf.jaxodraw.object.JaxoObjectFactory;
import net.sf.jaxodraw.object.loop.JaxoFLoop;
import net.sf.jaxodraw.util.JaxoColor;
import net.sf.jaxodraw.util.JaxoConstants;
import net.sf.jaxodraw.util.JaxoPrefs;


/** Tests JaxoVertexT5. */
public class JaxoVertexT5Test extends AbstractJaxo3PointObjectTest {
    private JaxoVertexT5 vertex = new JaxoVertexT5();

    /** Constructor: just calls super().
     * @param testName The name of this test.
     */
    public JaxoVertexT5Test(String testName) {
        super(testName);
    }

    /** JUnit setup: initialize the object. */
    @Override
    protected void setUp() {
        super.setUp();
        init();
    }

    private void init() {
        vertex.setLocation(50, 50, 100, 50, 100, 100);
        vertex.setColor(JaxoColor.BLACK);
        vertex.setFillColor(JaxoColor.BITTERSWEET);
        vertex.setStrokeWidth(1.f);
    }

    /** {@inheritDoc} */
    public JaxoObject getInstance() {
        return vertex;
    }

    /** Tests serialization. */
    public void testSerialization() {
        vertex.setStrokeWidth(1.5f);
        vertex.setColor(JaxoColor.GREEN);
        vertex.setFillColor(JaxoColor.GRAY);
        JaxoTestUtils.checkSerializedEquality(vertex);
    }

    /** Tests that a copy of an object is still the same object. */
    public void testCopy() {
        JaxoVertexT5 vertex2 = new JaxoVertexT5();
        vertex2.setState(vertex);

        vertex.setStrokeWidth(1.5f);
        vertex.setColor(JaxoColor.GREEN);
        vertex.setFillColor(JaxoColor.GRAY);
        JaxoTestUtils.checkCopyEquality(vertex);

        assertFalse(vertex2.isCopy(vertex));

        vertex2.setStrokeWidth(1.5f);
        vertex2.setColor(JaxoColor.GREEN);
        vertex2.setFillColor(JaxoColor.GRAY);
        assertTrue(vertex2.isCopy(vertex));

        JaxoFLoop loop = new JaxoFLoop();
        assertFalse(vertex.isCopy(loop));

    }

    /** Tests identity. */
    public void testIdentity() {
        JaxoVertexT5 vertex2 = vertex;
        assertEquals(vertex, vertex2);
    }

    /** Tests the latex command. */
    public void testLatexCommand() {
        Dimension dim = new Dimension(50, 50);

        vertex.setFillColor(JaxoColor.GRAYSCALE225);
        String expected = "\\GTri(50,0)(100,0)(100,-50){0.882}";
        String actual = vertex.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

        vertex.setFillColor(JaxoColor.GRAY);
        vertex.setColor(JaxoColor.GREEN);

        expected = "\\CTri(50,0)(100,0)(100,-50){Green}{Gray}";
        actual = vertex.latexCommand(1.f, dim);
        assertEquals("Latex double line commands do not match!", expected,
            actual);

        vertex.setFilled(false);
        expected = "\\ETri(50,0)(100,0)(100,-50)";
        actual = vertex.latexCommand(1.f, dim);
        assertEquals("Latex commands do not match!", expected, actual);

    }

    /** Tests axodraw. */
    public void testAxodraw() {
        JaxoAxodraw axo = new JaxoAxodraw();
        JaxoObject ob = axo.getJaxoObject("\\SetWidth{1.0}");
        assertNull(ob);
        String command;

        command = "\\GTri(50,0)(100,0)(100,-50){0.882}";
        vertex = (JaxoVertexT5) axo.getJaxoObject(command);
        checkVertexParameters(JaxoColor.BLACK, JaxoColor.GRAYSCALE225);

        command = "\\CTri(50,0)(100,0)(100,-50){Green}{Gray}";
        vertex = (JaxoVertexT5) axo.getJaxoObject(command);
        checkVertexParameters(JaxoColor.GREEN, JaxoColor.GRAY);

        command = "\\ETri(50,0)(100,0)(100,-50)";
        vertex = (JaxoVertexT5) axo.getJaxoObject(command);
        assertFalse("Vertex should not be filled!", vertex.isFilled());

    }

    private void checkVertexParameters(Color bordercolor, Color color) {
        assertEquals("VertexT5 doesn't have the right color!", bordercolor,
            vertex.getColor());
        assertEquals("VertexT5 doesn't have the right fill color!", color,
            vertex.getFillColor());
        assertTrue("VertexT5 doesn't have the right stroke!",
            vertex.strokeIs(1.f));
    }

    /** Tests the getBounds() method. */
    public void testGetBounds() {
        Rectangle bBox = vertex.getBounds();
        Rectangle expected = new Rectangle(48, 49, 53, 53);
        assertEquals("Bounding box not correct!", expected, bBox);
        vertex.setFilled(false);
        bBox = vertex.getBounds();
        assertEquals("Bounding box still not correct!", expected, bBox);

        assertNotNull(vertex.getBounds2D());
    }

    /** Tests createNewObject in JaxoObjectFactory. */
    public void testCreateNewObject() {
        Point p = new Point(50, 50);
        Point q = new Point(100, 50);
        Point r = new Point(100, 100);
        JaxoObject ob =
            JaxoObjectFactory.newObject(JaxoConstants.VERTEXT5, new Point[] {p, q, r});

        assertTrue("createNewObject did not return a VertexT5!",
            (ob instanceof JaxoVertexT5));

        vertex = (JaxoVertexT5) ob;
        assertEquals("VertexT5 doesn't have the right x1!", 50, vertex.getX());
        assertEquals("VertexT5 doesn't have the right y1!", 50, vertex.getY());
        assertEquals("VertexT5 doesn't have the right x2!", 100, vertex.getX2());
        assertEquals("VertexT5 doesn't have the right y2!", 50, vertex.getY2());
        assertEquals("VertexT5 doesn't have the right x!", 100, vertex.getX3());
        assertEquals("VertexT5 doesn't have the right y!", 100, vertex.getY3());
        assertEquals("VertexT5 doesn't have the right color!",
            JaxoColor.getColor(JaxoPrefs.getStringPref(JaxoPrefs.PREF_COLOR)),
            vertex.getColor());
        assertEquals("VertexT5 doesn't have the right fill color!",
            JaxoColor.GRAYSCALE225, vertex.getFillColor());
        assertTrue("VertexT5 doesn't have the right stroke!",
            vertex.strokeIs(JaxoPrefs.getFloatPref(JaxoPrefs.PREF_LINEWIDTH)));
    }

    /** Tests canBeSelected. */
    @Override
    public void testCanBeSelected() {
        vertex.setLocation(50, 50, 100, 50, 100, 100);

        int mode = JaxoConstants.MOVE;
        int handle = JaxoVertexT5.SELECT_P1;
        assertTrue("P1 in MOVE mode", vertex.canBeSelected(handle, mode));
        handle = JaxoVertexT5.SELECT_P2;
        assertTrue("P2 in MOVE mode", vertex.canBeSelected(handle, mode));
        handle = JaxoVertexT5.SELECT_P3;
        assertTrue("P3 in MOVE mode", vertex.canBeSelected(handle, mode));

        mode = JaxoConstants.RESIZE;
        handle = JaxoVertexT5.SELECT_P1;
        assertTrue("P1 in RESIZE mode", vertex.canBeSelected(handle, mode));
        handle = JaxoVertexT5.SELECT_P2;
        assertTrue("P2 in RESIZE mode", vertex.canBeSelected(handle, mode));
        handle = JaxoVertexT5.SELECT_P3;
        assertTrue("P3 in RESIZE mode", vertex.canBeSelected(handle, mode));
    }

    /** {@inheritDoc} */
    public String className() {
        return "JaxoVertexT5";
    }

    /** {@inheritDoc} */
    public void testJaxo20Compat() {
        JaxoVertexT5 object =
                (JaxoVertexT5) instanceFrom20OpenGraph(JaxoVertexT5.class);
        assertNotNull(object);
        check20Object("opened wrong object!", object);

        object = (JaxoVertexT5) instanceFrom20ImportGraph(JaxoVertexT5.class);
        assertNotNull(object);
        check20Object("imported wrong object!", object);
    }

    private void check20Object(String message, JaxoVertexT5 object) {
        assertEquals(message, 444, object.getX());
        assertEquals(message, 354, object.getY());
        assertEquals(message, 466, object.getX2());
        assertEquals(message, 342, object.getY2());
        assertEquals(message, 445, object.getX3());
        assertEquals(message, 329, object.getY3());

        assertTrue(message, object.strokeIs(1.4f));

        assertEquals(message, JaxoColor.BLACK, object.getColor());
        //assertEquals(message, JaxoColor.GRAYSCALE225, object.getFillColor());
        assertFalse(object.isFilled());
    }

    /** {@inheritDoc} */
    @Override
    public String[] serializableProperties() {
        return new String[] {
            "DLSeparation", "amp", "arrow", "arrowPosition", "color",
            "dash", "doubleLine", "fillColor", "filled", "flip",
            "paintArrow", "pointCount", "points", "strokeWidth", "wiggles"
        }; // 15
    }

    /** {@inheritDoc} */
    public void testArrowCoordinates() {
        // VertexT5 is a ParticleObject !?
    }
}
