/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.util;

import java.awt.Color;

import net.sf.jaxodraw.AbstractJaxoBaseTest;


/** JUnit based test for JaxoColor.java. */
public class JaxoColorTest extends AbstractJaxoBaseTest {
    /** List containing all the colors of JaxoColor. */
    private static Color[] colorList =
    {
        JaxoColor.GREEN_YELLOW, JaxoColor.YELLOW, JaxoColor.GOLDEN_ROD,
        JaxoColor.DANDELION, JaxoColor.APRICOT, JaxoColor.PEACH,
        JaxoColor.MELON, JaxoColor.YELLOW_ORANGE, JaxoColor.ORANGE,
        JaxoColor.BURNT_ORANGE, JaxoColor.BITTERSWEET, JaxoColor.RED_ORANGE,
        JaxoColor.MAHOGANY, JaxoColor.MAROON, JaxoColor.BRICK_RED,
        JaxoColor.RED, JaxoColor.ORANGE_RED, JaxoColor.RUBINE_RED,
        JaxoColor.WILDSTRAWBERRY, JaxoColor.SALMON, JaxoColor.CARNATION_PINK,
        JaxoColor.MAGENTA, JaxoColor.VIOLET_RED, JaxoColor.RHODAMINE,
        JaxoColor.MULBERRY, JaxoColor.RED_VIOLET, JaxoColor.FUCHSIA,
        JaxoColor.LAVENDER, JaxoColor.THISTLE, JaxoColor.ORCHID,
        JaxoColor.DARK_ORCHID, JaxoColor.PURPLE, JaxoColor.PLUM,
        JaxoColor.VIOLET, JaxoColor.ROYAL_PURPLE, JaxoColor.BLUE_VIOLET,
        JaxoColor.PERIWINKLE, JaxoColor.CADET_BLUE, JaxoColor.CORNFLOWER_BLUE,
        JaxoColor.MIDNIGHT_BLUE, JaxoColor.NAVY_BLUE, JaxoColor.ROYAL_BLUE,
        JaxoColor.BLUE, JaxoColor.CERULEAN, JaxoColor.CYAN,
        JaxoColor.PROCESS_BLUE, JaxoColor.SKY_BLUE, JaxoColor.TURQUOISE,
        JaxoColor.TEAL_BLUE, JaxoColor.AQUAMARINE, JaxoColor.BLUE_GREEN,
        JaxoColor.EMERALD, JaxoColor.JUNGLE_GREEN, JaxoColor.SEA_GREEN,
        JaxoColor.GREEN, JaxoColor.FOREST_GREEN, JaxoColor.PINE_GREEN,
        JaxoColor.LIME_GREEN, JaxoColor.YELLOW_GREEN, JaxoColor.SPRING_GREEN,
        JaxoColor.OLIVE_GREEN, JaxoColor.RAWSIENNA, JaxoColor.SEPIA,
        JaxoColor.BROWN, JaxoColor.TAN, JaxoColor.GRAY, JaxoColor.BLACK,
        JaxoColor.GRAYSCALE15, JaxoColor.GRAYSCALE30, JaxoColor.GRAYSCALE45,
        JaxoColor.GRAYSCALE60, JaxoColor.GRAYSCALE75, JaxoColor.GRAYSCALE90,
        JaxoColor.GRAYSCALE105, JaxoColor.GRAYSCALE120, JaxoColor.GRAYSCALE135,
        JaxoColor.GRAYSCALE150, JaxoColor.GRAYSCALE165, JaxoColor.GRAYSCALE180,
        JaxoColor.GRAYSCALE195, JaxoColor.GRAYSCALE210, JaxoColor.GRAYSCALE225,
        JaxoColor.GRAYSCALE240, JaxoColor.WHITE
    };

    /** The number of colors defined by JaxoColor. */
    private static int nofColors = colorList.length;

    /** List containing the names of all the colors of JaxoColor.
     *  These are the english names as defined by colordvi.sty and
     *  at the same time the keys for the translation into other languages.
     */
    private static String[] colorName =
    {
        "GreenYellow", "Yellow", "Goldenrod", "Dandelion", "Apricot", "Peach",
        "Melon", "YellowOrange", "Orange", "BurntOrange", "Bittersweet",
        "RedOrange", "Mahogany", "Maroon", "BrickRed", "Red", "OrangeRed",
        "RubineRed", "WildStrawberry", "Salmon", "CarnationPink", "Magenta",
        "VioletRed", "Rhodamine", "Mulberry", "RedViolet", "Fuchsia",
        "Lavender", "Thistle", "Orchid", "DarkOrchid", "Purple", "Plum",
        "Violet", "RoyalPurple", "BlueViolet", "Periwinkle", "CadetBlue",
        "CornflowerBlue", "MidnightBlue", "NavyBlue", "RoyalBlue", "Blue",
        "Cerulean", "Cyan", "ProcessBlue", "SkyBlue", "Turquoise", "TealBlue",
        "Aquamarine", "BlueGreen", "Emerald", "JungleGreen", "SeaGreen",
        "Green", "ForestGreen", "PineGreen", "LimeGreen", "YellowGreen",
        "SpringGreen", "OliveGreen", "RawSienna", "Sepia", "Brown", "Tan",
        "Gray", "Black", "GrayScale15", "GrayScale30", "GrayScale45",
        "GrayScale60", "GrayScale75", "GrayScale90", "GrayScale105",
        "GrayScale120", "GrayScale135", "GrayScale150", "GrayScale165",
        "GrayScale180", "GrayScale195", "GrayScale210", "GrayScale225",
        "GrayScale240", "White"
    };

    /** Constructor: just calls super().
     * @param testName The name of this test.
     */
    public JaxoColorTest(String testName) {
        super(testName);
    }

    /**
     * Test of getColor method.
     */
    public void testGetColor() {
        Color expResult;
        Color result;

        for (int i = 0; i < nofColors; i++) {
            expResult = colorList[i];
            result = JaxoColor.getColor(i);
            assertEquals("Colors do not match!", expResult, result);

            result = JaxoColor.getColor(colorName[i]);
            assertEquals("Colornames do not match!", expResult, result);
        }

        expResult = JaxoColor.BLACK;
        result = JaxoColor.getColor("My favorite color");
        assertEquals("Non-existent colors should return black!", expResult,
            result);

    }

    /**
     * Test of getColorName method.
     */
    public void testGetColorName() {
        String expResult;
        String result;

        for (int i = 0; i < nofColors; i++) {
            expResult = colorName[i];
            result = JaxoColor.getColorName(i);
            assertEquals("Colornames do not match!", expResult, result);

            result = JaxoColor.getColorName(colorList[i]);
            assertEquals("Colornames do not match!", expResult, result);

        }
    }

    /**
     * Test of getNofColors method.
     */
    public void testGetNofColors() {
        int expResult = nofColors;
        int result = JaxoColor.getColorCount();
        assertEquals("Mismatch in number of colors!", expResult, result);
    }

    /**
     * Test of isDefinedColorName methods.
     */
    public void testIsDefinedColorName() {
        // All defined color names are defined
        for (int i = 0; i < nofColors; i++) {
            assertTrue("isDefinedColorName(" + i + ")",
                JaxoColor.isDefinedColorName(JaxoColor.getColorName(i)));
        }

        // Undefined color names are not defined
        assertFalse("!isDefinedColorName",
            JaxoColor.isDefinedColorName("BlueAndRed"));

        // Non-gray scales are defined even in NO_GRAYSCALES mode
        for (int i = 0; i < JaxoColor.getColorCount(); i++) {
            if (!JaxoColor.isGrayScale(JaxoColor.getColor(i))) {
                assertTrue("isDefinedColorName(" + i + ")",
                    JaxoColor.isDefinedColorName(JaxoColor.getColorName(i),
                        JaxoColor.JAXO_COLORS_NO_GRAYSCALES_MODE));
            }
        }

        // Gray scales are not defined in NO_GRAYSCALES mode
        for (int i = 0; i < JaxoColor.getColorCount(); i++) {
            if (JaxoColor.isGrayScale(JaxoColor.getColor(i))) {
                assertFalse("!isDefinedColorName(" + i + ")",
                    JaxoColor.isDefinedColorName(JaxoColor.getColorName(i),
                        JaxoColor.JAXO_COLORS_NO_GRAYSCALES_MODE));
            }
        }

        // Any rgb-name is defined in ALL_COLORS_MODE
        assertTrue("isDefinedColorName",
            JaxoColor.isDefinedColorName("r235g12b65",
                JaxoColor.ALL_COLORS_MODE));
    }

    /**
     * Test of isDefinedColor methods.
     */
    public void testIsDefinedColor() {
        // All defined colors are defined
        for (int i = 0; i < nofColors; i++) {
            assertTrue("isDefinedColor(" + i + ")",
                JaxoColor.isDefinedColor(JaxoColor.getColor(i)));
        }

        // Undefined colors are not defined
        assertFalse("!isDefinedColor",
            JaxoColor.isDefinedColor(new Color(1, 1, 1)));

        // Non-gray scales are defined even in NO_GRAYSCALES mode
        for (int i = 0; i < JaxoColor.getColorCount(); i++) {
            if (!JaxoColor.isGrayScale(JaxoColor.getColor(i))) {
                assertTrue("isDefinedColor(" + i + ")",
                    JaxoColor.isDefinedColor(JaxoColor.getColor(i),
                        JaxoColor.JAXO_COLORS_NO_GRAYSCALES_MODE));
            }
        }

        // Gray scales are not defined in NO_GRAYSCALES mode
        for (int i = 0; i < JaxoColor.getColorCount(); i++) {
            if (JaxoColor.isGrayScale(JaxoColor.getColor(i))) {
                assertFalse("!isDefinedColor(" + i + ")",
                    JaxoColor.isDefinedColor(JaxoColor.getColor(i),
                        JaxoColor.JAXO_COLORS_NO_GRAYSCALES_MODE));
            }
        }

        // Any color is defined in ALL_COLORS_MODE
        assertTrue("isDefinedColor",
            JaxoColor.isDefinedColor(new Color(235, 12, 65),
                JaxoColor.ALL_COLORS_MODE));
    }

    /**
     * Test of getGrayScaleFloat method.
     */
    public void testGetGrayScaleFloat() {
        float expResult = 0.0f;
        float result = JaxoColor.getGrayScaleFloat(JaxoColor.ORANGE);
        assertEquals("Mismatch in getGrayScaleFloat!", expResult, result);
        expResult = 1.0f;
        result = JaxoColor.getGrayScaleFloat(JaxoColor.WHITE);
        assertEquals("Mismatch in getGrayScaleFloat!", expResult, result);
    }

    /**
     * Test of isGrayScale method.
     */
    public void testIsGrayScale() {
        boolean expResult;
        boolean result;
        for (int i = 0; i < nofColors; i++) {
            expResult = false;
            if ((i >= 67) && (i != 83)) {
                expResult = true;
            }
            result = JaxoColor.isGrayScale(colorList[i]);
            assertEquals("Mismatch in isGrayScale(color)!", expResult, result);
        }
    }

    /**
     * Test of getLatexColorCommand method.
     */
    public void testGetLatexColorCommand() {
        String expResult;
        String result;
        for (int i = 0; i < nofColors; i++) {
            expResult = "\\SetColor{" + colorName[i] + "}";
            if ((i > 66) && (i != 83)) {
                expResult = "\\SetColor{Gray}";
            }
            result = JaxoColor.getLatexColorCommand(colorList[i]);
            assertEquals("Mismatch in getLatexColorCommand(color)!",
                expResult, result);
        }
    }

    /**
     * Test of getGrayScaleColor method.
     */
    public void testGetGrayScaleColor() {
        float f = 0.0F;
        Color expResult = JaxoColor.BLACK;
        Color result = JaxoColor.getGrayScaleColor(f);
        assertEquals("Mismatch in getGrayScaleColor!", expResult, result);
        assertNull("getGrayScaleColor(f) does not return null for f>1!",
            JaxoColor.getGrayScaleColor(1.2f));
    }
}
