/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */

package net.sf.jaxodraw.util;

import java.util.Map;
import junit.framework.TestCase;

/**
 * Tests for JaxoPrefs.
 */
public class JaxoPrefsTest extends TestCase {
    private Map<String, Object> initialState;

    /**
     * Constructor.
     *
     * @param testName name of the test.
     */
    public JaxoPrefsTest(String testName) {
        super(testName);
    }

    /**
     * Setup. Save the current prefs to restore them after each test.
     *
     * @throws java.lang.Exception if any.
     */
    @Override
    protected void setUp() throws Exception {
        super.setUp();
        initialState = JaxoPrefs.getState();
    }

    /**
     * Teardown. Restore the saved prefs.
     *
     * @throws java.lang.Exception if any.
     */
    @Override
    protected void tearDown() throws Exception {
        super.tearDown();
        JaxoPrefs.setState(initialState);
    }

    /**
     * Test of get/setRecentFiles method, of class JaxoPrefs.
     */
    public void testGetSetRecentFiles() {
        JaxoPrefs.clearRecentFiles();
        String[] result = JaxoPrefs.getRecentFiles();
        assertEquals("", result[0]);

        JaxoPrefs.setRecentFiles(new String[] {"test"});
        result = JaxoPrefs.getRecentFiles();
        assertEquals("test", result[0]);
        assertEquals("", result[1]);

        JaxoPrefs.setRecentFiles(new String[] {"test", null});
        result = JaxoPrefs.getRecentFiles();
        assertEquals("test", result[0]);
        assertEquals("", result[1]);
    }

    /**
     * Test of clearRecentFiles method, of class JaxoPrefs.
     */
    public void testClearRecentFiles() {
        JaxoPrefs.clearRecentFiles();
        String[] recentFiles = JaxoPrefs.getRecentFiles();
        for (int i = 0; i < recentFiles.length; i++) {
            String recentFile = recentFiles[i];
            assertEquals(recentFile, "");
        }
    }

    /**
     * Test of addRecentFile method, of class JaxoPrefs.
     */
    public void testAddRecentFile() {
        JaxoPrefs.clearRecentFiles();
        JaxoPrefs.addRecentFile("test");
        String[] result = JaxoPrefs.getRecentFiles();
        assertEquals("test", result[0]);
        assertEquals("", result[1]);
    }

    /**
     * Test of get/setStringPref method, of class JaxoPrefs.
     */
    public void testGetSetStringPref() {
        JaxoPrefs.setStringPref(JaxoPrefs.PREF_EDITOR, "");
        assertEquals("", JaxoPrefs.getStringPref(JaxoPrefs.PREF_EDITOR));

        try {
            JaxoPrefs.getStringPref(JaxoPrefs.PREF_ARROW);
            fail("getStringPref should throw!");
        } catch (IllegalArgumentException e) {
            // expected
            assertNotNull(e);
        }
    }

    /**
     * Test of get/setIntPref method, of class JaxoPrefs.
     */
    public void testGetSetIntPref() {
        JaxoPrefs.setIntPref(JaxoPrefs.PREF_GRIDSIZE, 5);
        assertEquals(5, JaxoPrefs.getIntPref(JaxoPrefs.PREF_GRIDSIZE));

        try {
            JaxoPrefs.getIntPref(JaxoPrefs.PREF_ARROW);
            fail("getIntPref should throw!");
        } catch (IllegalArgumentException e) {
            // expected
            assertNotNull(e);
        }
    }

    /**
     * Test of get/setFloatPref method, of class JaxoPrefs.
     */
    public void testGetSetFloatPref() {
        JaxoPrefs.setFloatPref(JaxoPrefs.PREF_LINEWIDTH, 5.f);
        assertTrue(JaxoUtils.equal(
                5, JaxoPrefs.getFloatPref(JaxoPrefs.PREF_LINEWIDTH)));

        try {
            JaxoPrefs.getFloatPref(JaxoPrefs.PREF_ARROW);
            fail("getFloatPref should throw!");
        } catch (IllegalArgumentException e) {
            // expected
            assertNotNull(e);
        }
    }

    /**
     * Test of get/setBooleanPref method, of class JaxoPrefs.
     */
    public void testGetSetBooleanPref() {
        JaxoPrefs.setBooleanPref(JaxoPrefs.PREF_ARROW, true);
        assertTrue(JaxoPrefs.getBooleanPref(JaxoPrefs.PREF_ARROW));
        JaxoPrefs.setBooleanPref(JaxoPrefs.PREF_ARROW, false);
        assertFalse(JaxoPrefs.getBooleanPref(JaxoPrefs.PREF_ARROW));

        try {
            JaxoPrefs.getBooleanPref(JaxoPrefs.PREF_GRIDSIZE);
            fail("getBooleanPref should throw!");
        } catch (IllegalArgumentException e) {
            // expected
            assertNotNull(e);
        }
    }

    /**
     * Test of get/setState method, of class JaxoPrefs.
     */
    public void testGetState() {
        Map<String, Object> state = JaxoPrefs.getState();
        boolean arrow = JaxoPrefs.getBooleanPref(JaxoPrefs.PREF_ARROW);
        JaxoPrefs.setBooleanPref(JaxoPrefs.PREF_ARROW, !arrow);
        assertEquals(JaxoPrefs.getBooleanPref(JaxoPrefs.PREF_ARROW), !arrow);
        JaxoPrefs.setState(state);
        assertEquals(JaxoPrefs.getBooleanPref(JaxoPrefs.PREF_ARROW), arrow);
    }

    /**
     * Test of resetPref method, of class JaxoPrefs.
     */
    public void testResetPref() {
        JaxoPrefs.resetPref(JaxoPrefs.PREF_BROWSER);
        assertEquals("", JaxoPrefs.getStringPref(JaxoPrefs.PREF_BROWSER));
    }

}
