-- This script shows how to use the menubar and popupmenu controls from cplus.

local glu = glu()
local cv = canvas()
local gp = require "gplus"
local cp = require "cplus"

local cvwd = 500
local cvht = 300

local mbar              -- the menu bar
local mbarht = 28       -- height of menu bar
local popup             -- pop-up menu
local ticked = true     -- for toggling Ticked/Unticked item
local option = 1        -- for radio items in Options menu

--------------------------------------------------------------------------------

function ToggleTicked()
    ticked = not ticked
    mbar.tickitem(2, 3, ticked)
    if ticked then
        mbar.setitem(2, 3, "Ticked")
        glu.show("item is now Ticked")
    else
        mbar.setitem(2, 3, "Unticked")
        glu.show("item is now Unticked")
    end
end

--------------------------------------------------------------------------------

function SetOption(newoption)
    option = newoption
    mbar.radioitem(3, 1, option == 1)
    mbar.radioitem(3, 2, option == 2)
    mbar.radioitem(3, 3, option == 3)
    glu.show("Option "..option)
end

----------------------------------------------------------------------

function ChoosePopUpAction(x, y)
    popup.enableitem(4, false)
    popup.enableitem(6, false)

    popup.tickitem(5, true)
    popup.tickitem(6, true)

    -- yellow text on dark green translucent background
    popup.settextcolor(cp.yellow)
    popup.setbackcolor{0, 128, 0, 230}

    -- show pop-up menu at x,y and let user choose an action
    popup.show(x, y)
end

--------------------------------------------------------------------------------

function CreateCanvas()
    cv.create(cvwd, cvht)
    cv.position("middle")
    cv.rgba(cp.white)
    cv.fill()
    cv.font("default-bold", 12)
    cv.blend(2)
    cv.rgba(cp.black)
    local wd, ht = cv.text("clip", "Right-click to see a translucent pop-up menu.")
    local x = (cvwd - wd) // 2
    local y = (cvht - mbarht) // 2
    cv.paste("clip", x, y)
end

--------------------------------------------------------------------------------

function CreateMenus()
    cp.textshadowx = 2
    cp.textshadowy = 2

    -- create the menu bar
    mbar = cp.menubar()
    
    -- add some menus
    mbar.addmenu("File")
    mbar.addmenu("Edit")
    mbar.addmenu("Options")
    
    -- add items to File menu
    mbar.additem(1, "New", glu.show, {"New"})
    mbar.additem(1, "Open File...", glu.show, {"Open File"})
    mbar.additem(1, "Save File...", glu.show, {"Save File"})
    mbar.additem(1, "---", nil)
    mbar.additem(1, "Exit", glu.exit)
    
    -- add items to Edit menu
    mbar.additem(2, "Enabled", glu.show, {"Enabled"})
    mbar.additem(2, "Disabled", glu.show, {"Never see this"})
    mbar.additem(2, "Ticked", ToggleTicked)
    mbar.additem(2, "Ticked and Disabled", glu.show, {"Never see this"})
    
    -- add items to Options menu
    mbar.additem(3, "Option 1", SetOption, {1})
    mbar.additem(3, "Option 2", SetOption, {2})
    mbar.additem(3, "Option 3", SetOption, {3})

    -- create the pop-up menu
    popup = cp.popupmenu()
    popup.additem("Item 1", glu.show, {"Item 1"})
    popup.additem("Item 2", glu.show, {"Item 2"})
    popup.additem("---", nil)
    popup.additem("Disabled", glu.show, {"Never see this"})
    popup.additem("Always Ticked", glu.show, {"Always Ticked"})
    popup.additem("Ticked and Disabled", glu.show, {"Never see this"})
    popup.additem("---", nil)
    popup.additem("Exit", glu.exit)
end

--------------------------------------------------------------------------------

function ShowMenuBar()
    mbar.enableitem(2, 1, true)
    mbar.enableitem(2, 2, false)
    mbar.enableitem(2, 4, false)
    
    mbar.tickitem(2, 3, ticked)
    mbar.tickitem(2, 4, true)

    mbar.radioitem(3, 1, option == 1)
    mbar.radioitem(3, 2, option == 2)
    mbar.radioitem(3, 3, option == 3)

    mbar.show(0, 0, cvwd, mbarht)

    glu.update()
end

--------------------------------------------------------------------------------

function EventLoop()
    while true do
        local event = cp.process( glu.getevent() )
        -- event is an empty string if glu.getevent() was empty or if
        -- cp.process handled a click in the menu bar
        if #event == 0 then
            -- don't hog CPU if idle
            glu.sleep(5)
        elseif event:find("^cclick") then
            local _, x, y, button, mods = gp.split(event)
            x = tonumber(x)
            y = tonumber(y)
            if y > mbarht then
                -- show pop-up menu if user right-clicked
                if button == "right" and mods == "none" then
                    ChoosePopUpAction(x, y)
                end
            end
        else
            glu.doevent(event)
        end
    end
end

--------------------------------------------------------------------------------

function Main()
    glu.settitle("Menus")
    glu.setcolor("statusbar", 255, 255, 255)
    glu.show("The results of menu item selections will be shown here.")
    CreateCanvas()
    CreateMenus()
    ShowMenuBar()
    EventLoop()
end

--------------------------------------------------------------------------------

status, err = xpcall(Main, gp.trace)
if err then glu.continue(err) end
-- the following code is always executed;
-- note that cv.delete() does nothing if an error is detected before cv.create
cv.delete()
