/**
 * SweetFX by Christian Cann Schuldt Jensen ~ CeeJay.dk
 * Spherical Tonemap by Gilcher Pascal aka Marty McFly
 * Adaptive sharpen by bacondither
 * Gaussian Bloom by Ioxa
 * Mixed for Far Cry 6 Lite by CRubino
 */
 
 #include "ReShadeUI.fxh"
uniform float DarkLevel < __UNIFORM_SLIDER_FLOAT1
	ui_min = -0.400; ui_max = 0.400;
	ui_label = " Dark Level";
	ui_tooltip = "Adjust Darkness, minus for brighter and plus for darker";
> = 0.000;

uniform float LightIntensity < __UNIFORM_SLIDER_FLOAT1
	ui_min = -0.400; ui_max = 0.400;
	ui_label = " Light Intensity";
	ui_tooltip = "Adjust Light Intensity, plus for stronger Intensity and minus for weaker Intensity";
> = 0.000;


#define BlackPoint 0
#define WhitePoint 255 
#define HighlightClipping 0
#define Saturation 0.150
#define Exposure 0.000
#define Bleach 0.000
#define Defog 0.000
#define FogColor float3(0.000000,0.04000,0.090000)
#define sphericalAmount 0.000
#define curve_height 0.15000 
#define curveslope 0.500000
#define L_overshoot 0.003000
#define L_compr_low 0.167000
#define L_compr_high 0.334000
#define D_overshoot 0.009000
#define D_compr_low 0.450000
#define D_compr_high 0.700000
#define scale_lim 0.100000
#define scale_cs 0.056000
#define pm_p 0.700000
#define useAdaptiveSharpen 0

#ifndef fast_ops
	#define fast_ops 1 // Faster code path, small difference in quality
#endif

#include "ReShade.fxh"

texture AS_Pass0Tex { Width = BUFFER_WIDTH; Height = BUFFER_HEIGHT; Format = RG16F; };
sampler AS_Pass0Sampler { Texture = AS_Pass0Tex; };

// Helper funcs
#define sqr(a)         ( (a)*(a) )
#define max4(a,b,c,d)  ( max(max(a, b), max(c, d)) )

// Get destination pixel values
#define texc(x,y)      ( ReShade::PixelSize*float2(x, y) + tex )
#define getB(x,y)      ( saturate(tex2D(ReShade::BackBuffer, texc(x, y)).rgb) )
#define getT(x,y)      ( tex2D(AS_Pass0Sampler, texc(x, y)).xy )

// Soft if, fast linear approx
#define soft_if(a,b,c) ( saturate((a + b + c + 0.056)*rcp(abs(maxedge) + 0.03) - 0.85) )

// Soft limit, modified tanh
#if (fast_ops == 1) // Tanh approx
	#define soft_lim(v,s)  ( saturate(abs(v/s)*(27 + sqr(v/s))/(27 + 9*sqr(v/s)))*s )
#else
	#define soft_lim(v,s)  ( (exp(2*min(abs(v), s*24)/s) - 1)/(exp(2*min(abs(v), s*24)/s) + 1)*s )
#endif

// Weighted power mean
#define wpmean(a,b,w)  ( pow(abs(w)*pow(abs(a), pm_p) + abs(1-w)*pow(abs(b), pm_p), (1.0/pm_p)) )

// Component-wise distance
#define b_diff(pix)    ( abs(blur - c[pix]) )

// Fast-skip threshold, keep max possible luma error under 0.5/2^bit-depth
#if (fast_ops == 1)
	// Approx of x = tanh(x/y)*y + 0.5/2^bit-depth, y = min(L_overshoot, D_overshoot)
	#define min_overshoot  ( min(abs(L_overshoot), abs(D_overshoot)) )
	#define fskip_th       ( 0.114*pow(min_overshoot, 0.676) + 3.20e-4 ) // 10-bits
	//#define fskip_th       ( 0.045*pow(min_overshoot, 0.667) + 1.75e-5 ) // 14-bits
#else
	// x = tanh(x/y)*y + 0.5/2^bit-depth, y = 0.0001
	#define fskip_th       ( 0.000110882 ) // 14-bits
#endif

// Smoothstep to linearstep approx
//#define SStLS(a,b,x,c) ( clamp(-(6*(c - 1)*(b - x))/(5*(a - b)) - 0.1*c + 1.1, c, 1) )

// Center pixel diff
#define mdiff(a,b,c,d,e,f,g) ( abs(luma[g] - luma[a]) + abs(luma[g] - luma[b])       \
                             + abs(luma[g] - luma[c]) + abs(luma[g] - luma[d])       \
                             + 0.5*(abs(luma[g] - luma[e]) + abs(luma[g] - luma[f])) )

float2 AdaptiveSharpenP0(float4 vpos : SV_Position, float2 tex : TEXCOORD) : SV_Target
{
	// Get points and clip out of range values (BTB & WTW)
	// [                c9                ]
	// [           c1,  c2,  c3           ]
	// [      c10, c4,  c0,  c5, c11      ]
	// [           c6,  c7,  c8           ]
	// [                c12               ]
	float3 c[13] = { getB( 0, 0), getB(-1,-1), getB( 0,-1), getB( 1,-1), getB(-1, 0),
	                 getB( 1, 0), getB(-1, 1), getB( 0, 1), getB( 1, 1), getB( 0,-2),
	                 getB(-2, 0), getB( 2, 0), getB( 0, 2) };

	// Colour to luma, fast approx gamma, avg of rec. 709 & 601 luma coeffs
	float luma = sqrt(dot(float3(0.2558, 0.6511, 0.0931), sqr(c[0])));

	// Blur, gauss 3x3
	float3 blur = (2*(c[2]+c[4]+c[5]+c[7]) + (c[1]+c[3]+c[6]+c[8]) + 4*c[0])/16;

	// Contrast compression, center = 0.5, scaled to 1/3
	float c_comp = saturate(4.0/15.0 + 0.9*exp2(dot(blur, -37.0/15.0)));

	// Edge detection
	// Relative matrix weights
	// [          1          ]
	// [      4,  5,  4      ]
	// [  1,  5,  6,  5,  1  ]
	// [      4,  5,  4      ]
	// [          1          ]
	float edge = length( 1.38*(b_diff(0))
	                   + 1.15*(b_diff(2) + b_diff(4)  + b_diff(5)  + b_diff(7))
	                   + 0.92*(b_diff(1) + b_diff(3)  + b_diff(6)  + b_diff(8))
	                   + 0.23*(b_diff(9) + b_diff(10) + b_diff(11) + b_diff(12)) );

	return float2(edge*c_comp, luma);
}

float3 AdaptiveSharpenP1(float4 vpos : SV_Position, float2 tex : TEXCOORD) : SV_Target
{
	float3 origsat = getB(0, 0);

	// Get texture points, .x = edge, .y = luma
	// [                d22               ]
	// [           d24, d9,  d23          ]
	// [      d21, d1,  d2,  d3, d18      ]
	// [ d19, d10, d4,  d0,  d5, d11, d16 ]
	// [      d20, d6,  d7,  d8, d17      ]
	// [           d15, d12, d14          ]
	// [                d13               ]
	float2 d[25] = { getT( 0, 0), getT(-1,-1), getT( 0,-1), getT( 1,-1), getT(-1, 0),
	                 getT( 1, 0), getT(-1, 1), getT( 0, 1), getT( 1, 1), getT( 0,-2),
	                 getT(-2, 0), getT( 2, 0), getT( 0, 2), getT( 0, 3), getT( 1, 2),
	                 getT(-1, 2), getT( 3, 0), getT( 2, 1), getT( 2,-1), getT(-3, 0),
	                 getT(-2, 1), getT(-2,-1), getT( 0,-3), getT( 1,-2), getT(-1,-2) };

	// Allow for higher overshoot if the current edge pixel is surrounded by similar edge pixels
	float maxedge = max4( max4(d[1].x,d[2].x,d[3].x,d[4].x), max4(d[5].x,d[6].x,d[7].x,d[8].x),
	                      max4(d[9].x,d[10].x,d[11].x,d[12].x), d[0].x );

	// [          x          ]
	// [       z, x, w       ]
	// [    z, z, x, w, w    ]
	// [ y, y, y, 0, y, y, y ]
	// [    w, w, x, z, z    ]
	// [       w, x, z       ]
	// [          x          ]
	float sbe = soft_if(d[2].x,d[9].x, d[22].x)*soft_if(d[7].x,d[12].x,d[13].x)  // x dir
	          + soft_if(d[4].x,d[10].x,d[19].x)*soft_if(d[5].x,d[11].x,d[16].x)  // y dir
	          + soft_if(d[1].x,d[24].x,d[21].x)*soft_if(d[8].x,d[14].x,d[17].x)  // z dir
	          + soft_if(d[3].x,d[23].x,d[18].x)*soft_if(d[6].x,d[20].x,d[15].x); // w dir

	#if (fast_ops == 1)
		float2 cs = lerp( float2(L_compr_low,  D_compr_low),
		                  float2(L_compr_high, D_compr_high), saturate(1.091*sbe - 2.282) );
	#else
		float2 cs = lerp( float2(L_compr_low,  D_compr_low),
		                  float2(L_compr_high, D_compr_high), smoothstep(2, 3.1, sbe) );
	#endif

	float luma[25] = { d[0].y,  d[1].y,  d[2].y,  d[3].y,  d[4].y,
	                   d[5].y,  d[6].y,  d[7].y,  d[8].y,  d[9].y,
	                   d[10].y, d[11].y, d[12].y, d[13].y, d[14].y,
	                   d[15].y, d[16].y, d[17].y, d[18].y, d[19].y,
	                   d[20].y, d[21].y, d[22].y, d[23].y, d[24].y };

	// Pre-calculated default squared kernel weights
	const float3 W1 = float3(0.5,           1.0, 1.41421356237); // 0.25, 1.0, 2.0
	const float3 W2 = float3(0.86602540378, 1.0, 0.54772255751); // 0.75, 1.0, 0.3

	// Transition to a concave kernel if the center edge val is above thr
	#if (fast_ops == 1)
		float3 dW = sqr(lerp( W1, W2, saturate(2.4*d[0].x - 0.82) ));
	#else
		float3 dW = sqr(lerp( W1, W2, smoothstep(0.3, 0.8, d[0].x) ));
	#endif

	float mdiff_c0 = 0.02 + 3*( abs(luma[0]-luma[2]) + abs(luma[0]-luma[4])
	                          + abs(luma[0]-luma[5]) + abs(luma[0]-luma[7])
	                          + 0.25*(abs(luma[0]-luma[1]) + abs(luma[0]-luma[3])
	                                 +abs(luma[0]-luma[6]) + abs(luma[0]-luma[8])) );

	// Use lower weights for pixels in a more active area relative to center pixel area
	// This results in narrower and less visible overshoots around sharp edges
	float weights[12] = { ( min(mdiff_c0/mdiff(24, 21, 2,  4,  9,  10, 1),  dW.y) ),   // c1
	                      ( dW.x ),                                                    // c2
	                      ( min(mdiff_c0/mdiff(23, 18, 5,  2,  9,  11, 3),  dW.y) ),   // c3
	                      ( dW.x ),                                                    // c4
	                      ( dW.x ),                                                    // c5
	                      ( min(mdiff_c0/mdiff(4,  20, 15, 7,  10, 12, 6),  dW.y) ),   // c6
	                      ( dW.x ),                                                    // c7
	                      ( min(mdiff_c0/mdiff(5,  7,  17, 14, 12, 11, 8),  dW.y) ),   // c8
	                      ( min(mdiff_c0/mdiff(2,  24, 23, 22, 1,  3,  9),  dW.z) ),   // c9
	                      ( min(mdiff_c0/mdiff(20, 19, 21, 4,  1,  6,  10), dW.z) ),   // c10
	                      ( min(mdiff_c0/mdiff(17, 5,  18, 16, 3,  8,  11), dW.z) ),   // c11
	                      ( min(mdiff_c0/mdiff(13, 15, 7,  14, 6,  8,  12), dW.z) ) }; // c12

	weights[0] = (max(max((weights[8]  + weights[9])/4,  weights[0]), 0.25) + weights[0])/2;
	weights[2] = (max(max((weights[8]  + weights[10])/4, weights[2]), 0.25) + weights[2])/2;
	weights[5] = (max(max((weights[9]  + weights[11])/4, weights[5]), 0.25) + weights[5])/2;
	weights[7] = (max(max((weights[10] + weights[11])/4, weights[7]), 0.25) + weights[7])/2;

	// Calculate the negative part of the laplace kernel and the low threshold weight
	float lowthrsum   = 0;
	float weightsum   = 0;
	float neg_laplace = 0;

	[unroll] for (int pix = 0; pix < 12; ++pix)
	{
		#if (fast_ops == 1)
			float lowthr = clamp((13.2*d[pix + 1].x - 0.221), 0.01, 1);

			neg_laplace += sqr(luma[pix + 1])*(weights[pix]*lowthr);
		#else
			float t = saturate((d[pix + 1].x - 0.01)/0.09);
			float lowthr = t*t*(2.97 - 1.98*t) + 0.01; // t*t*(3 - a*3 - (2 - a*2)*t) + a

			neg_laplace += pow(abs(luma[pix + 1]) + 0.06, 2.4)*(weights[pix]*lowthr);
		#endif
		weightsum += weights[pix]*lowthr;
		lowthrsum += lowthr/12;
	}

	#if (fast_ops == 1)
		neg_laplace = sqrt(neg_laplace/weightsum);
	#else
		neg_laplace = pow(abs(neg_laplace/weightsum), (1.0/2.4)) - 0.06;
	#endif

	// Compute sharpening magnitude function
	float sharpen_val = curve_height/(curve_height*curveslope*pow(abs(d[0].x), 3.5) + 0.625);

	// Calculate sharpening diff and scale
	float sharpdiff = (d[0].y - neg_laplace)*(lowthrsum*sharpen_val + 0.01);

	// Skip limiting on flat areas where sharpdiff is low
	[branch] if (abs(sharpdiff) > fskip_th)
	{
		// Calculate local near min & max, partial sort
		// Manually unrolled outer loop, solves OpenGL slowdown
		{
			float temp; int i; int ii;

			// 1st iteration
			[unroll] for (i = 0; i < 24; i += 2)
			{
				temp = luma[i];
				luma[i]   = min(luma[i], luma[i+1]);
				luma[i+1] = max(temp, luma[i+1]);
			}
			[unroll] for (ii = 24; ii > 0; ii -= 2)
			{
				temp = luma[0];
				luma[0]    = min(luma[0], luma[ii]);
				luma[ii]   = max(temp, luma[ii]);

				temp = luma[24];
				luma[24]   = max(luma[24], luma[ii-1]);
				luma[ii-1] = min(temp, luma[ii-1]);
			}

			// 2nd iteration
			[unroll] for (i = 1; i < 23; i += 2)
			{
				temp = luma[i];
				luma[i]   = min(luma[i], luma[i+1]);
				luma[i+1] = max(temp, luma[i+1]);
			}
			[unroll] for (ii = 23; ii > 1; ii -= 2)
			{
				temp = luma[1];
				luma[1]    = min(luma[1], luma[ii]);
				luma[ii]   = max(temp, luma[ii]);

				temp = luma[23];
				luma[23]   = max(luma[23], luma[ii-1]);
				luma[ii-1] = min(temp, luma[ii-1]);
			}

			#if (fast_ops != 1) // 3rd iteration
				[unroll] for (i = 2; i < 22; i += 2)
				{
					temp = luma[i];
					luma[i]   = min(luma[i], luma[i+1]);
					luma[i+1] = max(temp, luma[i+1]);
				}
				[unroll] for (ii = 22; ii > 2; ii -= 2)
				{
					temp = luma[2];
					luma[2]    = min(luma[2], luma[ii]);
					luma[ii]   = max(temp, luma[ii]);

					temp = luma[22];
					luma[22]   = max(luma[22], luma[ii-1]);
					luma[ii-1] = min(temp, luma[ii-1]);
				}
			#endif
		}

		// Calculate tanh scale factors
		#if (fast_ops == 1)
			float nmax = (max(luma[23], d[0].y)*2 + luma[24])/3;
			float nmin = (min(luma[1],  d[0].y)*2 + luma[0])/3;

			float min_dist  = min(abs(nmax - d[0].y), abs(d[0].y - nmin));
			float pos_scale = min_dist + L_overshoot;
			float neg_scale = min_dist + D_overshoot;
		#else
			float nmax = (max(luma[22] + luma[23]*2, d[0].y*3) + luma[24])/4;
			float nmin = (min(luma[2]  + luma[1]*2,  d[0].y*3) + luma[0])/4;

			float min_dist  = min(abs(nmax - d[0].y), abs(d[0].y - nmin));
			float pos_scale = min_dist + min(L_overshoot, 1.0001 - min_dist - d[0].y);
			float neg_scale = min_dist + min(D_overshoot, 0.0001 + d[0].y - min_dist);
		#endif

		pos_scale = min(pos_scale, scale_lim*(1 - scale_cs) + pos_scale*scale_cs);
		neg_scale = min(neg_scale, scale_lim*(1 - scale_cs) + neg_scale*scale_cs);

		// Soft limited anti-ringing with tanh, wpmean to control compression slope
		sharpdiff = wpmean( max(sharpdiff, 0), soft_lim( max(sharpdiff, 0), pos_scale ), cs.x )
		          - wpmean( min(sharpdiff, 0), soft_lim( min(sharpdiff, 0), neg_scale ), cs.y );
	}

	// Compensate for saturation loss/gain while making pixels brighter/darker
	float sharpdiff_lim = saturate(d[0].y + sharpdiff) - d[0].y;
	float satmul = (d[0].y + max(sharpdiff_lim*0.9, sharpdiff_lim)*1.03 + 0.03)/(d[0].y + 0.03);
	float3 res = d[0].y + (sharpdiff_lim*3 + sharpdiff)/4 + (origsat - d[0].y)*satmul;

	return saturate(res);
}

float3 LevelsPass(float4 vpos : SV_Position, float2 texcoord : TexCoord) : SV_Target
{
	float black_point_float = BlackPoint / 255.0;
	float white_point_float = WhitePoint == BlackPoint ? (255.0 / 0.00025) : (255.0 / (WhitePoint - BlackPoint)); // Avoid division by zero if the white and black point are the same

	float3 color = tex2D(ReShade::BackBuffer, texcoord).rgb;
	color = color * white_point_float - (black_point_float *  white_point_float);

	if (HighlightClipping)
	{
		float3 clipped_colors;

		clipped_colors = any(color > saturate(color)) // any colors whiter than white?
			? float3(1.0, 0.0, 0.0)
			: color;
		clipped_colors = all(color > saturate(color)) // all colors whiter than white?
			? float3(1.0, 1.0, 0.0)
			: clipped_colors;
		clipped_colors = any(color < saturate(color)) // any colors blacker than black?
			? float3(0.0, 0.0, 1.0)
			: clipped_colors;
		clipped_colors = all(color < saturate(color)) // all colors blacker than black?
			? float3(0.0, 1.0, 1.0)
			: clipped_colors;

		color = clipped_colors;
	}

	return color;
}


#define RGB_Gain float3(1.040000+LightIntensity,1.040000+LightIntensity,1.070000+LightIntensity)
#define RGB_Gamma float3(1.100000,1.120000,1.100000)
#define RGB_Lift float3(0.930000-DarkLevel,0.930000-DarkLevel,0.880000-DarkLevel)
#define Gamma 1.000

float3 TonemapPass(float4 position : SV_Position, float2 texcoord : TexCoord) : SV_Target
{
	float3 color = tex2D(ReShade::BackBuffer, texcoord).rgb;
	color = saturate(color - Defog * FogColor * 2.55); // Defog
	color *= pow(2.0f, Exposure); // Exposure
	color = pow(color, Gamma); // Gamma

	const float3 coefLuma = float3(0.2126, 0.7152, 0.0722);
	float lum = dot(coefLuma, color);
	
	float L = saturate(10.0 * (lum - 0.45));
	float3 A2 = Bleach * color;

	float3 result1 = 2.0f * color * lum;
	float3 result2 = 1.0f - 2.0f * (1.0f - lum) * (1.0f - color);
	
	float3 newColor = lerp(result1, result2, L);
	float3 mixRGB = A2 * newColor;
	color += ((1.0f - A2) * mixRGB);
	
	float3 middlegray = dot(color, (1.0 / 3.0));
	float3 diffcolor = color - middlegray;
	color = (color + diffcolor * Saturation) / (1 + (diffcolor * Saturation)); // Saturation
	
	return color;
}

float3 NyukNyang(float4 position : SV_Position, float2 texcoord : TexCoord) : SV_Target
{
	float3 color = tex2D(ReShade::BackBuffer, texcoord).rgb;
	
	// -- Lift --
	color = color * (1.5 - 0.5 * RGB_Lift) + 0.5 * RGB_Lift - 0.5;
	color = saturate(color); // Is not strictly necessary, but does not cost performance
	
	// -- Gain --
	color *= RGB_Gain; 
	
	// -- Gamma --
	color = pow(abs(color), 1.0 / RGB_Gamma);
	
	return saturate(color);
}


#define Amount 1.500000
#define Center float2(0.500000,0.500000)
#define Radius 2.000000
#define Ratio 1.000000
#define Slope 2
#define Type 0
float4 VignettePass(float4 vpos : SV_Position, float2 tex : TexCoord) : SV_Target
{
	float4 color = tex2D(ReShade::BackBuffer, tex);

	if (Type == 0)
	{
		// Set the center
		float2 distance_xy = tex - Center;

		// Adjust the ratio
		distance_xy *= float2((BUFFER_RCP_HEIGHT / BUFFER_RCP_WIDTH), Ratio);

		// Calculate the distance
		distance_xy /= Radius;
		float distance = dot(distance_xy, distance_xy);

		// Apply the vignette
		color.rgb *= (1.0 + pow(distance, Slope * 0.5) * Amount); //pow - multiply
	}

	if (Type == 1) // New round (-x*x+x) + (-y*y+y) method.
	{
		tex = -tex * tex + tex;
		color.rgb = saturate(((BUFFER_RCP_HEIGHT / BUFFER_RCP_WIDTH)*(BUFFER_RCP_HEIGHT / BUFFER_RCP_WIDTH) * Ratio * tex.x + tex.y) * 4.0) * color.rgb;
	}

	if (Type == 2) // New (-x*x+x) * (-y*y+y) TV style method.
	{
		tex = -tex * tex + tex;
		color.rgb = saturate(tex.x * tex.y * 100.0) * color.rgb;
	}
		
	if (Type == 3)
	{
		tex = abs(tex - 0.5);
		float tc = dot(float4(-tex.x, -tex.x, tex.x, tex.y), float4(tex.y, tex.y, 1.0, 1.0)); //XOR

		tc = saturate(tc - 0.495);
		color.rgb *= (pow((1.0 - tc * 200), 4) + 0.25); //or maybe abs(tc*100-1) (-(tc*100)-1)
	}
  
	if (Type == 4)
	{
		tex = abs(tex - 0.5);
		float tc = dot(float4(-tex.x, -tex.x, tex.x, tex.y), float4(tex.y, tex.y, 1.0, 1.0)); //XOR

		tc = saturate(tc - 0.495) - 0.0002;
		color.rgb *= (pow((1.0 - tc * 200), 4) + 0.0); //or maybe abs(tc*100-1) (-(tc*100)-1)
	}

	if (Type == 5) // MAD version of 2
	{
		tex = abs(tex - 0.5);
		float tc = tex.x * (-2.0 * tex.y + 1.0) + tex.y; //XOR

		tc = saturate(tc - 0.495);
		color.rgb *= (pow((-tc * 200 + 1.0), 4) + 0.25); //or maybe abs(tc*100-1) (-(tc*100)-1)
		//color.rgb *= (pow(((tc*200.0)-1.0),4)); //or maybe abs(tc*100-1) (-(tc*100)-1)
	}

	if (Type == 6) // New round (-x*x+x) * (-y*y+y) method.
	{
		//tex.y /= float2((BUFFER_RCP_HEIGHT / BUFFER_RCP_WIDTH), Ratio);
		float tex_xy = dot(float4(tex, tex), float4(-tex, 1.0, 1.0)); //dot is actually slower
		color.rgb = saturate(tex_xy * 4.0) * color.rgb;
	}

	return color;
}


//Bloom Settings
#if !defined GaussBloomTexScale
	#define GaussBloomTexScale 0.5
#endif

#define GaussianBloomRadius 2
#define GaussianBloomOffset 2.00
#define Threshold 0.800
#define BloomTint float3(1.0,1.0,2.0)
#define Exposure2 35.0
#define GaussianBloomSaturation 0.500
#define DitherStrength 1.00
#define GaussianBloomStrength 0.20
#define NearCutoff 0.0
#define FarCutoff 1.0
#define BlendIfRange 0.25	
#define DebugMode 0

texture GaussianBloomTex { Width = BUFFER_WIDTH*GaussBloomTexScale; Height = BUFFER_HEIGHT*GaussBloomTexScale; Format = RGBA8; };
texture GaussianBloomTex2 { Width = BUFFER_WIDTH*GaussBloomTexScale; Height = BUFFER_HEIGHT*GaussBloomTexScale; Format = RGBA8; };

sampler GaussianBloomSampler { Texture = GaussianBloomTex; AddressW = BORDER; AddressU = BORDER; AddressV = BORDER;};
sampler GaussianBloomSampler2 { Texture = GaussianBloomTex2; AddressW = BORDER; AddressU = BORDER; AddressV = BORDER;};

#define Offset02y ReShade::PixelSize.y*1.1824255238
#define Offset03y ReShade::PixelSize.y*3.0293122308
#define Offset04y ReShade::PixelSize.y*5.0040701377

#define Offset02x ReShade::PixelSize.x*1.1824255238
#define Offset03x ReShade::PixelSize.x*3.0293122308
#define Offset04x ReShade::PixelSize.x*5.0040701377

#define OffsetA2y ReShade::PixelSize.y*1.4584295168
#define OffsetA3y ReShade::PixelSize.y*3.40398480678
#define OffsetA4y ReShade::PixelSize.y*5.3518057801
#define OffsetA5y ReShade::PixelSize.y*7.302940716
#define OffsetA6y ReShade::PixelSize.y*9.2581597095

#define OffsetA2x ReShade::PixelSize.x*1.4584295168
#define OffsetA3x ReShade::PixelSize.x*3.40398480678
#define OffsetA4x ReShade::PixelSize.x*5.3518057801
#define OffsetA5x ReShade::PixelSize.x*7.302940716
#define OffsetA6x ReShade::PixelSize.x*9.2581597095

#define OffsetB2y 1.4895848401*ReShade::PixelSize.y
#define OffsetB3y 3.4757135714*ReShade::PixelSize.y
#define OffsetB4y 5.4618796741*ReShade::PixelSize.y
#define OffsetB5y 7.4481042327*ReShade::PixelSize.y
#define OffsetB6y 9.4344079746*ReShade::PixelSize.y
#define OffsetB7y 11.420811147*ReShade::PixelSize.y
#define OffsetB8y 13.4073334*ReShade::PixelSize.y
#define OffsetB9y 15.3939936778*ReShade::PixelSize.y
#define OffsetB10y 17.3808101174*ReShade::PixelSize.y
#define OffsetB11y 19.3677999584*ReShade::PixelSize.y

#define OffsetB2x 1.4895848401*ReShade::PixelSize.x
#define OffsetB3x 3.4757135714*ReShade::PixelSize.x
#define OffsetB4x 5.4618796741*ReShade::PixelSize.x
#define OffsetB5x 7.4481042327*ReShade::PixelSize.x
#define OffsetB6x 9.4344079746*ReShade::PixelSize.x
#define OffsetB7x 11.420811147*ReShade::PixelSize.x
#define OffsetB8x 13.4073334*ReShade::PixelSize.x
#define OffsetB9x 15.3939936778*ReShade::PixelSize.x
#define OffsetB10x 17.3808101174*ReShade::PixelSize.x
#define OffsetB11x 19.3677999584*ReShade::PixelSize.x

#define OffsetC2y ReShade::PixelSize.y*1.4953705027
#define OffsetC3y ReShade::PixelSize.y*3.4891992113
#define OffsetC4y ReShade::PixelSize.y*5.4830312105
#define OffsetC5y ReShade::PixelSize.y*7.4768683759
#define OffsetC6y ReShade::PixelSize.y*9.4707125766
#define OffsetC7y ReShade::PixelSize.y*11.4645656736
#define OffsetC8y ReShade::PixelSize.y*13.4584295168
#define OffsetC9y ReShade::PixelSize.y*15.4523059431
#define OffsetC10y ReShade::PixelSize.y*17.4461967743
#define OffsetC11y ReShade::PixelSize.y*19.4401038149
#define OffsetC12y ReShade::PixelSize.y*21.43402885
#define OffsetC13y ReShade::PixelSize.y*23.4279736431
#define OffsetC14y ReShade::PixelSize.y*25.4219399344
#define OffsetC15y ReShade::PixelSize.y*27.4159294386

#define OffsetC2x ReShade::PixelSize.x*1.4953705027
#define OffsetC3x ReShade::PixelSize.x*3.4891992113
#define OffsetC4x ReShade::PixelSize.x*5.4830312105
#define OffsetC5x ReShade::PixelSize.x*7.4768683759
#define OffsetC6x ReShade::PixelSize.x*9.4707125766
#define OffsetC7x ReShade::PixelSize.x*11.4645656736
#define OffsetC8x ReShade::PixelSize.x*13.4584295168
#define OffsetC9x ReShade::PixelSize.x*15.4523059431
#define OffsetC10x ReShade::PixelSize.x*17.4461967743
#define OffsetC11x ReShade::PixelSize.x*19.4401038149
#define OffsetC12x ReShade::PixelSize.x*21.43402885
#define OffsetC13x ReShade::PixelSize.x*23.4279736431
#define OffsetC14x ReShade::PixelSize.x*25.4219399344
#define OffsetC15x ReShade::PixelSize.x*27.4159294386

#define OffsetD2y ReShade::PixelSize.y*1.4953705027
#define OffsetD3y ReShade::PixelSize.y*3.4891992113
#define OffsetD4y ReShade::PixelSize.y*5.4830312105
#define OffsetD5y ReShade::PixelSize.y*7.4768683759
#define OffsetD6y ReShade::PixelSize.y*9.4707125766
#define OffsetD7y ReShade::PixelSize.y*11.4645656736
#define OffsetD8y ReShade::PixelSize.y*13.4584295168
#define OffsetD9y ReShade::PixelSize.y*15.4523059431
#define OffsetD10y ReShade::PixelSize.y*17.4461967743
#define OffsetD11y ReShade::PixelSize.y*19.4661974725
#define OffsetD12y ReShade::PixelSize.y*21.4627427973
#define OffsetD13y ReShade::PixelSize.y*23.4592916956
#define OffsetD14y ReShade::PixelSize.y*25.455844494
#define OffsetD15y ReShade::PixelSize.y*27.4524015179
#define OffsetD16y ReShade::PixelSize.y*29.4489630909
#define OffsetD17y ReShade::PixelSize.y*31.445529535
#define OffsetD18y ReShade::PixelSize.y*33.4421011704

#define OffsetD2x ReShade::PixelSize.x*1.4953705027
#define OffsetD3x ReShade::PixelSize.x*3.4891992113
#define OffsetD4x ReShade::PixelSize.x*5.4830312105
#define OffsetD5x ReShade::PixelSize.x*7.4768683759
#define OffsetD6x ReShade::PixelSize.x*9.4707125766
#define OffsetD7x ReShade::PixelSize.x*11.4645656736
#define OffsetD8x ReShade::PixelSize.x*13.4584295168
#define OffsetD9x ReShade::PixelSize.x*15.4523059431
#define OffsetD10x ReShade::PixelSize.x*17.4461967743
#define OffsetD11x ReShade::PixelSize.x*19.4661974725
#define OffsetD12x ReShade::PixelSize.x*21.4627427973
#define OffsetD13x ReShade::PixelSize.x*23.4592916956
#define OffsetD14x ReShade::PixelSize.x*25.455844494
#define OffsetD15x ReShade::PixelSize.x*27.4524015179
#define OffsetD16x ReShade::PixelSize.x*29.4489630909
#define OffsetD17x ReShade::PixelSize.x*31.445529535
#define OffsetD18x ReShade::PixelSize.x*33.4421011704

float3 GaussianBloomFinal(in float4 pos : SV_Position, in float2 texcoord : TEXCOORD) : COLOR
{

	float3 blur = tex2D(GaussianBloomSampler, texcoord / GaussianBloomOffset).rgb;

	blur.rgb = lerp(dot(blur.rgb,0.333333),blur.rgb,GaussianBloomSaturation);

	if(DitherStrength)
	{
	float sine = sin(dot(texcoord, float2(12.9898,78.233)));
	float noise = frac(sine * 43758.5453 + texcoord.x) * 0.012 - 0.006;
	blur.rgb -= (float3(-noise, noise, -noise)*DitherStrength);
	blur.rgb = saturate(blur.rgb);
	}

	float3 orig = tex2D(ReShade::BackBuffer, texcoord).rgb;
	orig = lerp(orig, (1.0 - ((1.0 - orig) * (1.0 - blur))), GaussianBloomStrength);

	if(DebugMode == 1)
	{
	orig = blur;
	}
	else
	{
		if(DebugMode == 2)
		{

		float Depth = tex2D(ReShade::DepthBuffer, texcoord).r;
		float Near = (100-NearCutoff)*0.01;
		float Far = FarCutoff*0.01;
		float mask = 1.0;
		float range;

		range = Near*BlendIfRange;
		float mix = 1-Depth;
		mask -= smoothstep(Near-(range),Near+(range),mix);

		range = Far*BlendIfRange;
		mask = lerp(mask,0.0,smoothstep(Far-(range),Far+(range),Depth));

		orig = float3(mask,mask,mask);
		}
	}
	
	return orig;
	
}

float3 GaussianBloomBrightPass(in float4 pos : SV_Position, in float2 texcoord : TEXCOORD) : COLOR
{
	
	float3 color = tex2D( ReShade::BackBuffer, texcoord * GaussianBloomOffset).rgb;
	
	color.rgb *= ( 1.0f + ( color.rgb / ( Threshold * Threshold )) );
	color.rgb *= Exposure2;
    color.rgb -= (5.0f);

    color.rgb = max( color.rgb, 0.0f );

    color.rgb /= ( 10.0 + color.rgb );

	color.rgb *= BloomTint;
	
	if( NearCutoff || FarCutoff < 100.0)
	{
		float Depth = tex2D(ReShade::DepthBuffer, texcoord * GaussianBloomOffset).r;
		float Near = (100-NearCutoff)*0.01;
		float Far = FarCutoff*0.01;
		float mask = 1.0;
		float range;
					
		if(NearCutoff)
		{
			range = Near*BlendIfRange;
			float mix = 1-Depth;
			mask -= smoothstep(Near-(range),Near+(range),mix);
		}
					
		if(FarCutoff < 100.0)
		{
			range = Far*BlendIfRange;
			mask = lerp(mask,0.0,smoothstep(Far-(range),Far+(range),Depth));
		}
			
		color.rgb *= mask;
	}
		
	return color;
}

float3 GaussianBloom1(in float4 pos : SV_Position, in float2 texcoord : TEXCOORD) : COLOR
{

float3 blur = tex2D(GaussianBloomSampler, texcoord).rgb;
	
if(GaussianBloomRadius == 2)	
{
	static const float offset[11] = { 0.0, OffsetB2y, OffsetB3y, OffsetB4y, OffsetB5y, OffsetB6y, OffsetB7y, OffsetB8y, OffsetB9y, OffsetB10y, OffsetB11y };
	static const float weight[11] = { 0.06649, 0.1284697563, 0.111918249, 0.0873132676, 0.0610011113, 0.0381655709, 0.0213835661, 0.0107290241, 0.0048206869, 0.0019396469, 0.0006988718 };
	
	blur *= weight[0];
	
	[loop]
	for(int i = 1; i < 11; ++i)
	{
		
		float2 coord = float2(0.0, offset[i]);
		
		blur += tex2Dlod(GaussianBloomSampler, float4(texcoord + coord, 0.0, 0.0)).rgb * weight[i];
		blur += tex2Dlod(GaussianBloomSampler, float4(texcoord - coord, 0.0, 0.0)).rgb * weight[i];
	}
}	
else
{
	if(GaussianBloomRadius == 3)	
	{
		static const float offset[15] = { 0.0, OffsetC2y, OffsetC3y, OffsetC4y, OffsetC5y, OffsetC6y, OffsetC7y, OffsetC8y, OffsetC9y, OffsetC10y, OffsetC11y, OffsetC12y, OffsetC13y, OffsetC14y, OffsetC15y };
		static const float weight[15] = { 0.0443266667, 0.0872994708, 0.0820892038, 0.0734818355, 0.0626171681, 0.0507956191, 0.0392263968, 0.0288369812, 0.0201808877, 0.0134446557, 0.0085266392, 0.0051478359, 0.0029586248, 0.0016187257, 0.0008430913 };
	
		blur *= weight[0];
	
		[loop]
		for(int i = 1; i < 15; ++i)
		{
			float2 coord = float2(0.0, offset[i]);
		
			blur += tex2Dlod(GaussianBloomSampler, float4(texcoord + coord, 0.0, 0.0)).rgb * weight[i];
			blur += tex2Dlod(GaussianBloomSampler, float4(texcoord - coord, 0.0, 0.0)).rgb * weight[i];
		}
	}
	else
	{
		if(GaussianBloomRadius == 4)	
		{
			static const float offset[18] = { 0.0, OffsetD2y, OffsetD3y, OffsetD4y, OffsetD5y, OffsetD6y, OffsetD7y, OffsetD8y, OffsetD9y, OffsetD10y, OffsetD11y, OffsetD12y, OffsetD13y, OffsetD14y, OffsetD15y, OffsetD16y, OffsetD17y, OffsetD18y };
			static const float weight[18] = { 0.033245, 0.0659162217, 0.0636705814, 0.0598194658, 0.0546642566, 0.0485871646, 0.0420045997, 0.0353207015, 0.0288880982, 0.0229808311, 0.0177815511, 0.013382297, 0.0097960001, 0.0069746748, 0.0048301008, 0.0032534598, 0.0021315311, 0.0013582974 };
	
			blur *= weight[0];
	
			[loop]
			for(int i = 1; i < 18; ++i)
			{
				float2 coord = float2(0.0, offset[i]);
		
				blur += tex2Dlod(GaussianBloomSampler, float4(texcoord + coord, 0.0, 0.0)).rgb * weight[i];
				blur += tex2Dlod(GaussianBloomSampler, float4(texcoord - coord, 0.0, 0.0)).rgb * weight[i];
			}
		}
		else 
		{
			if(GaussianBloomRadius == 1)	
			{
				static const float offset[6] = { 0.0, OffsetA2y, OffsetA3y, OffsetA4y, OffsetA5y, OffsetA6y };
				static const float weight[6] = { 0.13298, 0.23227575, 0.1353261595, 0.0511557427, 0.01253922, 0.0019913644 };
	
				blur *= weight[0];
	
				[loop]
				for(int i = 1; i < 6; ++i)
				{
					float2 coord = float2(0.0, offset[i]);
		
					blur += tex2Dlod(GaussianBloomSampler, float4(texcoord + coord, 0.0, 0.0)).rgb * weight[i];
					blur += tex2Dlod(GaussianBloomSampler, float4(texcoord - coord, 0.0, 0.0)).rgb * weight[i];
				}
			}
			else 
			{
				if(GaussianBloomRadius == 0)	
				{
					static const float offset[4] = { 0.0, Offset02y, Offset03y, Offset04y };
					static const float weight[4] = { 0.39894, 0.2959599993, 0.0045656525, 0.00000149278686458842 };
		
					blur *= weight[0];
	
					[loop]
					for(int i = 1; i < 4; ++i)
					{
						float2 coord = float2(0.0, offset[i]);
		
						blur += tex2Dlod(GaussianBloomSampler, float4(texcoord + coord, 0.0, 0.0)).rgb * weight[i];
						blur += tex2Dlod(GaussianBloomSampler, float4(texcoord - coord, 0.0, 0.0)).rgb * weight[i];
					}
				}
			}	
			
		}
	}
}	

	return (blur);
}

float3 GaussianBloom2(in float4 pos : SV_Position, in float2 texcoord : TEXCOORD) : COLOR
{

float3 blur = tex2D(GaussianBloomSampler2, texcoord).rgb;

if(GaussianBloomRadius == 2)	
{
	static const float offset[11] = { 0.0, OffsetB2x, OffsetB3x, OffsetB4x, OffsetB5x, OffsetB6x, OffsetB7x, OffsetB8x, OffsetB9x, OffsetB10x, OffsetB11x };
	static const float weight[11] = { 0.06649, 0.1284697563, 0.111918249, 0.0873132676, 0.0610011113, 0.0381655709, 0.0213835661, 0.0107290241, 0.0048206869, 0.0019396469, 0.0006988718 };
	
	blur *= weight[0];
	
	[loop]
	for(int i = 1; i < 11; ++i)
	{
		
		float2 coord = float2(offset[i],0.0);
		
		blur += tex2Dlod(GaussianBloomSampler2, float4(texcoord + coord, 0.0, 0.0)).rgb * weight[i];
		blur += tex2Dlod(GaussianBloomSampler2, float4(texcoord - coord, 0.0, 0.0)).rgb * weight[i];
	}
}	
else
{
	if(GaussianBloomRadius == 3)	
	{
		static const float offset[15] = { 0.0, OffsetC2x, OffsetC3x, OffsetC4x, OffsetC5x, OffsetC6x, OffsetC7x, OffsetC8x, OffsetC9x, OffsetC10x, OffsetC11x, OffsetC12x, OffsetC13x, OffsetC14x, OffsetC15x };
		static const float weight[15] = { 0.0443266667, 0.0872994708, 0.0820892038, 0.0734818355, 0.0626171681, 0.0507956191, 0.0392263968, 0.0288369812, 0.0201808877, 0.0134446557, 0.0085266392, 0.0051478359, 0.0029586248, 0.0016187257, 0.0008430913 };
	
		blur *= weight[0];
	
		[loop]
		for(int i = 1; i < 15; ++i)
		{
			float2 coord = float2(offset[i],0.0);
		
			blur += tex2Dlod(GaussianBloomSampler2, float4(texcoord + coord, 0.0, 0.0)).rgb * weight[i];
			blur += tex2Dlod(GaussianBloomSampler2, float4(texcoord - coord, 0.0, 0.0)).rgb * weight[i];
		}
	}
	else
	{
		if(GaussianBloomRadius == 4)	
		{
			static const float offset[18] = { 0.0, OffsetD2x, OffsetD3x, OffsetD4x, OffsetD5x, OffsetD6x, OffsetD7x, OffsetD8x, OffsetD9x, OffsetD10x, OffsetD11x, OffsetD12x, OffsetD13x, OffsetD14x, OffsetD15x, OffsetD16x, OffsetD17x, OffsetD18x };
			static const float weight[18] = { 0.033245, 0.0659162217, 0.0636705814, 0.0598194658, 0.0546642566, 0.0485871646, 0.0420045997, 0.0353207015, 0.0288880982, 0.0229808311, 0.0177815511, 0.013382297, 0.0097960001, 0.0069746748, 0.0048301008, 0.0032534598, 0.0021315311, 0.0013582974 };
	
			blur *= weight[0];
	
			[loop]
			for(int i = 1; i < 18; ++i)
			{
				float2 coord = float2(offset[i],0.0);
		
				blur += tex2Dlod(GaussianBloomSampler2, float4(texcoord + coord, 0.0, 0.0)).rgb * weight[i];
				blur += tex2Dlod(GaussianBloomSampler2, float4(texcoord - coord, 0.0, 0.0)).rgb * weight[i];
			}
		}
		else 
		{
			if(GaussianBloomRadius == 1)	
			{
				static const float offset[6] = { 0.0, OffsetA2x, OffsetA3x, OffsetA4x, OffsetA5x, OffsetA6x };
				static const float weight[6] = { 0.13298, 0.23227575, 0.1353261595, 0.0511557427, 0.01253922, 0.0019913644 };
	
				blur *= weight[0];
	
				[loop]
				for(int i = 1; i < 6; ++i)
				{
					float2 coord = float2(offset[i],0.0);
		
					blur += tex2Dlod(GaussianBloomSampler2, float4(texcoord + coord, 0.0, 0.0)).rgb * weight[i];
					blur += tex2Dlod(GaussianBloomSampler2, float4(texcoord - coord, 0.0, 0.0)).rgb * weight[i];
				}
			}
			else 
			{
				if(GaussianBloomRadius == 0)	
				{
					static const float offset[4] = { 0.0, Offset02x, Offset03x, Offset04x };
					static const float weight[4] = { 0.39894, 0.2959599993, 0.0045656525, 0.00000149278686458842 };
		
					blur *= weight[0];
	
					[loop]
					for(int i = 1; i < 4; ++i)
					{
						float2 coord = float2(offset[i],0.0);
		
						blur += tex2Dlod(GaussianBloomSampler2, float4(texcoord + coord, 0.0, 0.0)).rgb * weight[i];
						blur += tex2Dlod(GaussianBloomSampler2, float4(texcoord - coord, 0.0, 0.0)).rgb * weight[i];
					}
				}
			}	
			
		}
	}
}	
	return (blur);
}


float3 SphericalPass(float4 position : SV_Position, float2 texcoord : TexCoord) : SV_Target
{
	float3 color = tex2D(ReShade::BackBuffer, texcoord).rgb;
	
	float3 signedColor = color.rgb * 2.0 - 1.0;
	float3 sphericalColor = sqrt(1.0 - signedColor.rgb * signedColor.rgb);
	sphericalColor = sphericalColor * 0.5 + 0.5;
	sphericalColor *= color.rgb;
	color.rgb += sphericalColor.rgb * sphericalAmount;
	color.rgb *= 0.95;
	
	return color.rgb;
}

technique Volumetric_Booster
{
	pass BrightPass
	{
		VertexShader = PostProcessVS;
		PixelShader = GaussianBloomBrightPass;
		RenderTarget = GaussianBloomTex;
	}
		
	pass Bloom1
	{
		VertexShader = PostProcessVS;
		PixelShader = GaussianBloom1;
		RenderTarget = GaussianBloomTex2;
	}
	
	pass Bloom2
	{
		VertexShader = PostProcessVS;
		PixelShader = GaussianBloom2;
		RenderTarget = GaussianBloomTex;
	}

	pass Bloom
	{
		VertexShader = PostProcessVS;
		PixelShader = GaussianBloomFinal;
	}
	
	pass Tonemap {VertexShader = PostProcessVS;PixelShader = TonemapPass;}
	
}

technique FC6
{
	
	//pass Levels {VertexShader = PostProcessVS; PixelShader = LevelsPass; }
	
	pass LighterBrighter
	{
		VertexShader = PostProcessVS;
		PixelShader = NyukNyang;
	}
			
	//pass Tonemap {VertexShader = PostProcessVS;PixelShader = TonemapPass;}
	
	#if useAdaptiveSharpen == 1
	pass AdaptiveSharpenPass1
	{
		VertexShader = PostProcessVS;
		PixelShader  = AdaptiveSharpenP0;
		RenderTarget = AS_Pass0Tex;
	}

	pass AdaptiveSharpenPass2
	{
		VertexShader = PostProcessVS;
		PixelShader  = AdaptiveSharpenP1;
	}
	#endif 
	
		
	//pass SphericalTonemap {VertexShader = PostProcessVS; PixelShader = SphericalPass;}
	pass {VertexShader = PostProcessVS; PixelShader = VignettePass;}
	
}




	
