local function is_point_in_poly(x, z, poly)
    local num = #poly
    local i = 0
    local j = num
    local c = false
    for i = 1, num do
        if ((poly[i][2] > z) ~= (poly[j][2] > z)) and
            (x < (poly[i][1] + (poly[j][1] - poly[i][1]) * (z - poly[i][2]) /
                    (poly[j][2] - poly[i][2]))) then
            c = not c
        end
        j = i
    end
    return c
end

local function closest_point_in_segment(Px, Pz, Ax, Az, Bx, Bz)
    local ABx, ABz = Bx - Ax, Bz - Az
    local APx, APz = Px - Ax, Pz - Az
    local prod = ABx * APx + ABz * APz

    if prod <= 0 then
		return Ax, Az, 0
	end
    local alpha = prod / (ABx * ABx + ABz * ABz)
    if alpha >= 1 then
		return Bx, Bz, 1
	end
    return Ax + alpha * ABx, Az + alpha * ABz, alpha
end

local function closest_point_in_poly(x0, z0, offset_x, offset_z, vertices)
	local x, z = x0 - offset_x, z0 - offset_z
	local idx0 = #vertices
	local min_dist, min_x, min_z, idxa, idxb
    for idx1 = 1, #vertices do
        local x0, z0 = vertices[idx0][1], vertices[idx0][2]
		local x1, z1 = vertices[idx1][1], vertices[idx1][2]
		local pt_x, pt_z, alpha = closest_point_in_segment(x, z, x0, z0, x1, z1)
		local dist = VecUtil_Length(x - pt_x, z - pt_z)
		if min_dist == nil or (min_dist and dist < min_dist) then
			min_dist, min_x, min_z = dist, pt_x, pt_z
			idxa = alpha < 1 and idx0 or nil
			idxb = alpha > 0 and idx1 or nil
		end
		idx0 = idx1
	end
	return min_x + offset_x, min_z + offset_z, idxa, idxb
end

local function calc_bissector_vector(A, O, B)
    --bisector of angle AOB
    local OAx, OAz = A[1] - O[1], A[2] - O[2]
    local OBx, OBz = B[1] - O[1], B[2] - O[2]
    local a1 = math.atan2(OAz, OAx)
    local a2 = math.atan2(OBz, OBx)
    local a = (a1 + a2) / 2
    if a1 > a2 then
        a = a + math.pi
    end
    local n = (math.sin((a2 - a1) / 2) ~= 0) and 1 / math.abs(math.sin((a2 - a1) / 2)) or 0
    return n, math.cos(a), math.sin(a), 2 * PI - (a2 - a1) % (2 * PI)
end

function rotate_vertices(vertices, angle)
	--angle in radian
	local new_vertices = {}
	for i = 1, #vertices do
		table.insert(new_vertices, {VecUtil_RotateDir(vertices[i][1], vertices[i][2], angle)})
	end
	return new_vertices
end

local function subdivise_segment(v0, v1, min_step)
	local subdivision = {}
    local x0, z0 = v0[1], v0[2]
	local x1, z1 = v1[1], v1[2]
	local length = VecUtil_Length(x0 - x1, z0 - z1)
	local num = math.max(math.ceil(length / min_step - 1), 1)
	for k = 0, num - 1 do
		local pos_x, pos_z = VecUtil_Lerp(x0, z0, x1, z1, k / num)
		table.insert(subdivision, {pos_x, pos_z})
	end
	return subdivision
end

local function subdivise(vertices, min_step)
	local subdivision = {}
	local idx0 = #vertices
    for idx1 = 1, #vertices do
		for _, s in pairs(subdivise_segment(vertices[idx0], vertices[idx1], min_step)) do
			table.insert(subdivision, s)			
		end
		idx0 = idx1
	end
	return subdivision
end

local function vertices_bissectors(vertices)
    local bissectors = {}
    for i = 1, #vertices do
        table.insert(bissectors, {calc_bissector_vector(vertices[(i - 2) % #vertices + 1], vertices[i], vertices[i % #vertices + 1])})
    end
    return bissectors
end

local function cut_edges(vertices, bissectors, l)
	--cut edge so plant fit in them
	local bad_indexes = {}
	local new_vertices = {}
	local cuts = {}
	for i = 1, #vertices do
        local x, z = unpack(vertices[i])
		local angle = bissectors[i][4]
		if angle < 0.001 then --too edgy
			table.insert(bad_indexes, {(i - 2) % #vertices + 1, i})
			table.insert(bad_indexes, {i, i % #vertices + 1})
		elseif angle < PI / 2 - 1E-3 then --need a cut
			local d = l / math.tan(angle)
			local x1, z1 = unpack(vertices[(i - 2) % #vertices + 1])
			local n1x, n1z, length1 = VecUtil_NormalAndLength(x1 - x, z1 - z)
			local x2, z2 = unpack(vertices[i % #vertices + 1])
			local n2x, n2z, length2 = VecUtil_NormalAndLength(x2 - x, z2 - z)
	
			if d >= length1 / 2 then --too edgy
				table.insert(bad_indexes, {(i - 2) % #vertices + 1, i})
			end
			if d >= length2 / 2 then
				table.insert(bad_indexes, {i, i % #vertices + 1})
			end
			table.insert(new_vertices, {x + n1x * d, z + n1z * d})
			table.insert(new_vertices, {x + n2x * d, z + n2z * d})
			table.insert(cuts, false)
			table.insert(cuts, true)
		else
			table.insert(new_vertices, {x, z})
			table.insert(cuts, false)
		end
	end
	if #bad_indexes > 0 then
		return false, bad_indexes
	else
		return true, cuts, new_vertices 
	end
end

local function cut_reflex_angle(vertices, bissectors, l)
	--cut shifed vertices to fit calc_edge_plant for grid plant
	local new_vertices = {}
	local index_mapping = {}
	for i = 1, #vertices do
		local x, z = unpack(vertices[i])
		local angle = bissectors[i][4]
		if angle > 3 * PI / 2 + 1E-3 then
			local x1, z1 = unpack(vertices[(i - 2) % #vertices + 1])
			local n1x, n1z, length1 = VecUtil_NormalAndLength(x1 - x, z1 - z)
			local x2, z2 = unpack(vertices[i % #vertices + 1])
			local n2x, n2z, length2 = VecUtil_NormalAndLength(x2 - x, z2 - z)
			local d = (-1 / math.tan(angle / 2) - 1) * math.abs(l)
			table.insert(new_vertices, {x + n1x * d, z + n1z * d})
			table.insert(new_vertices, {x + n2x * d, z + n2z * d})
			table.insert(index_mapping, i)
			table.insert(index_mapping, i)
		else
			table.insert(new_vertices, {x , z})
			table.insert(index_mapping, i)
		end
	end
	return new_vertices, index_mapping
end

local function calc_edge_plant(vertices, bissectors, cuts, l, min_step)
	local segments = {}
	local idx0 = #vertices
	for idx1 = 1, #vertices do
		if not cuts[idx1] then
			local x0, z0 = vertices[idx0][1], vertices[idx0][2]
			local x1, z1 = vertices[idx1][1], vertices[idx1][2]
			local nx, nz = VecUtil_NormalAndLength(x1 - x0, z1 - z0)

			local angle = bissectors[idx0][4]
			if angle > PI + 1E-3 then
				local d = math.min(-l / math.tan(angle / 2), l)
				x0, z0 = x0 - nx * d, z0 - nz * d
			end
			angle = bissectors[idx1][4]
			if angle > PI + 1E-3 then
				local d = math.min(-l / math.tan(angle / 2), l)
				x1, z1 = x1 + nx * d, z1 + nz * d
			end

			local sub = subdivise_segment({x0, z0}, {x1, z1}, min_step)
			table.insert(sub, {x1, z1})
			for i = 1, #sub - 1 do
				local x0, z0 = unpack(sub[i])
				local x1, z1 = unpack(sub[i + 1])
				table.insert(segments, {x0, z0, x1, z1})
			end
		end
		idx0 = idx1
	end
	return segments
end

function shift_vertices(vertices, bissectors, d, limit)
	local shifted_vertices = {}	
	for i = 1, #vertices do
		local x, z = unpack(vertices[i])
		local n, nx, nz = unpack(bissectors[i])
		if limit then --avoid going to far, usefull for player_collison in reflex angle
			n = math.min(n , limit)
		end
		table.insert(shifted_vertices, {x + d * n * nx, z + d * n * nz})
	end
	return shifted_vertices
end

local function calc_sections(z, poly)
	--return intersection between poly and line of abscissa z
    local num = #poly
    local i = 0
    local j = num
    local sections = {}
    for i = 1, num do
        if ((poly[i][2] > z) ~= (poly[j][2] > z)) then
			table.insert(sections, poly[i][1] + (poly[j][1] - poly[i][1]) * (z - poly[i][2]) / (poly[j][2] - poly[i][2]))
		end
        j = i
    end

	table.sort(sections)
    return sections
end

local function make_cut(x0, x1, step, rand)
	--cut randomly into piece of size [step/2, setp)
	if x1 - x0 < step then
		return {{(x0 + x1) / 2, x1 - x0}} --mid and length
	else
		local cut = x0 + step / 2 + (x1 - x0 - step) * (rand and math.random() or 0.5)
		local left, right = make_cut(x0, cut, step, rand), make_cut(cut, x1, step, rand)
		for _,v in ipairs(right) do 
			table.insert(left, v)
		end
		return left
	end
end

local function make_grid(vertices, base_step_x, base_step_z, random)
	local grid = {}

	local min_z, max_z = vertices[1][2], vertices[1][2]
	for k, v in pairs(vertices) do
		if v[2] > max_z then max_z = v[2] end
		if v[2] < min_z then min_z = v[2] end	
	end
	local n_step_z = math.ceil((max_z - min_z) / base_step_z)
	local step_z = (max_z - min_z) / n_step_z

	for idx_z = 0, n_step_z-1 do
		local z = min_z + (idx_z + 0.5) * step_z
		local sections = calc_sections(z, vertices)
		
		for idx_s = 1, #sections / 2 do
			for _, s in ipairs(make_cut(sections[2 * idx_s - 1], sections[2 * idx_s], base_step_x, random)) do
				table.insert(grid, {s[1], z, s[2], step_z})
			end
		end
	end
	return grid
end

local function centroid(vertices)
	if #vertices == 1 then
		return vertices[1][1], vertices[1][2], 0
	elseif #vertices == 2 then
		return (vertices[1][1] + vertices[2][1])/2, (vertices[1][2] + vertices[2][2])/2, 0
	else
		local cx, cz, A = 0, 0, 0
		local idx0 = #vertices
		for idx1 = 1, #vertices do
		    local x0, z0, x1, z1 = vertices[idx0][1], vertices[idx0][2], vertices[idx1][1], vertices[idx1][2]
		    cx = cx + (x0 + x1) * (x0 * z1 - x1 * z0)
		    cz = cz + (z0 + z1) * (x0 * z1 - x1 * z0)
		    A = A + (x0 * z1 - x1 * z0) / 2 
		    idx0 = idx1
		end
		if A ~= 0 then
		    return cx / 6 / A, cz / 6 / A, A
		end
	end
end

local function vectorial_product(A, B, P)
	--angle ABP   
    return (B[1] - A[1]) * (P[2] - A[2]) - (B[2] - A[2]) * (P[1] - A[1])
end

local function line_line_intersection(v1, v2, v3, v4)
	local x1, z1, x2, z2, x3, z3, x4, z4 = v1[1], v1[2], v2[1], v2[2], v3[1], v3[2], v4[1], v4[2]
    local d = (x1 - x2) * (z3 - z4) - (z1 - z2) * (x3 - x4)
    if math.abs(d) < 1e-15 then
        return math.abs(vectorial_product(v1, v2, v3)) < 1E-15
    end
    local nx = (x1 * z2 - z1 * x2) * (x3 - x4) - (x1 - x2) * (x3 * z4 - z3 * x4)
    local nz = (x1 * z2 - z1 * x2) * (z3 - z4) - (z1 - z2) * (x3 * z4 - z3 * x4)
    return {nx / d, nz / d}
end

local function is_between(A, B, C)
	local Ax, Az, Bx, Bz, Cx, Cz = A[1], A[2], B[1], B[2], C[1], C[2]
    if math.abs(Ax - Cx) < 1e-15 then  --use x or z ?
        return (Az <= Bz and Bz <= Cz) or (Az >= Bz and Bz >= Cz)
    else
        return (Ax <= Bx and Bx <= Cx) or (Ax >= Bx and Bx >= Cx)
    end
end

local function segment_segment_intersection(v1, v2, v3, v4)
    local i = line_line_intersection(v1, v2, v3, v4)
    if i == false then --parallel segment, not aligned
        return false
    end
    if i == true then --parallel aligned segment
        if is_between(v1, v3, v2) then return true end
        if is_between(v1, v4, v2) then return true end
        if is_between(v3, v1, v4) then return true end
        if is_between(v3, v2, v4) then return true end
        return false
    end
    return is_between(v1, i, v2) and is_between(v3, i, v4)
end

local function crossing_edge(vertices)
	local bad_indexes = {}
	for i = 1, #vertices - 2 do
		for j = i + 2, math.min(#vertices - 2 + i, #vertices) do --to avoid neighbor
			if segment_segment_intersection(vertices[i], vertices[i + 1], vertices[j], vertices[j % #vertices + 1]) then
				table.insert(bad_indexes, {i, i + 1})
				table.insert(bad_indexes, {j, j % #vertices + 1})
			end
		end
	end
	return bad_indexes
end

local function CalcRadius(vertices)
	local radius = 0
	for i = 1, #vertices do
		local length = VecUtil_Length(vertices[i][1], vertices[i][2])
		if length > radius then
			radius = length
		end
	end
	return radius
end

-- export all local functions
local env = {}
local i = 1
while true do
    local name, value = debug.getlocal(1, i)
    if name == 'env' then break end
    env[name] = value
    i = i + 1
end
return env
