-- import all mathutil
local env = getfenv(1)
for k, v in pairs(require('boat_mathutil')) do env[k] = v end
setfenv(1, env)
--grab some functions from modutil
local env = {}
require("modutil")["InsertPostInitFunctions"](env, false)
env.modname = KnownModIndex:GetModActualName("New Boat Shapes")
local AddRecipe = env.AddRecipe
local GetModConfigData = env.GetModConfigData

local assets =
{
}

local item_assets =
{
}

local prefabs =
{
	"mast",
	"burnable_locator_medium",
	"steeringwheel",
	"rudder",
	"boat",
	"boatlip",
	"boatlip_fx",
	"boat_water_fx",
	"boat_leak",
	"fx_boat_crackle",
	"boatfragment03",
	"boatfragment04",
	"boatfragment05",
	"fx_boat_pop",
	"boat_player_collision_common",
	"boat_item_collision_common",
	"boat_hull_collision_common",
	"boat_world_collision_common",
	"walkingplank",
	"ground_support",
	"boat_plant"
}

local item_prefabs = {}

local function OnRepaired(inst)
	inst.SoundEmitter:PlaySound("turnoftides/common/together/boat/repair_with_wood")
end

local function BoatCam_IsEnabledFn()
	return Profile:IsBoatCameraEnabled()
end

local function BoatCam_ActiveFn(params, parent, best_dist_sq)
	local state = params.updater.state
	local tpos = params.target:GetPosition()
	state.last_platform_x, state.last_platform_z = tpos.x, tpos.z

	local pan_gain, heading_gain, distance_gain = TheCamera:GetGains()
	TheCamera:SetGains(1.5, heading_gain, distance_gain)
end

local function BoatCam_UpdateFn(dt, params, parent, best_dist_sq)
	local tpos = params.target:GetPosition()

	local state = params.updater.state
	local platform_x, platform_y, platform_z = tpos:Get()

	local velocity_x = dt == 0 and 0 or ((platform_x - state.last_platform_x) / dt)
	local velocity_z = dt == 0 and 0 or ((platform_z - state.last_platform_z) / dt)
	local velocity_normalized_x, velocity_normalized_z = 0, 0
	local velocity = 0
	local min_velocity = 0.4
	local velocity_sq = velocity_x * velocity_x + velocity_z * velocity_z

	if velocity_sq >= min_velocity * min_velocity then
		velocity = math.sqrt(velocity_sq)
		velocity_normalized_x = velocity_x / velocity
		velocity_normalized_z = velocity_z / velocity
		velocity = math.max(velocity - min_velocity, 0)
	end

	local look_ahead_max_dist = 5
	local look_ahead_max_velocity = 3
	local look_ahead_percentage = math.min(math.max(velocity / look_ahead_max_velocity, 0), 1)
	local look_ahead_amount = look_ahead_max_dist * look_ahead_percentage

	--Average target_camera_offset to get rid of some of the noise.
	state.target_camera_offset.x = (state.target_camera_offset.x + velocity_normalized_x * look_ahead_amount) / 2
	state.target_camera_offset.z = (state.target_camera_offset.z + velocity_normalized_z * look_ahead_amount) / 2

	state.last_platform_x, state.last_platform_z = platform_x, platform_z

	local camera_offset_lerp_speed = 0.25
	state.camera_offset.x, state.camera_offset.z = VecUtil_Lerp(state.camera_offset.x, state.camera_offset.z, state.target_camera_offset.x, state.target_camera_offset.z, dt * camera_offset_lerp_speed)

	TheCamera:SetOffset(state.camera_offset + (tpos - parent:GetPosition()))

	local pan_gain, heading_gain, distance_gain = TheCamera:GetGains()
	local pan_lerp_speed = 0.75
	pan_gain = Lerp(pan_gain, state.target_pan_gain, dt * pan_lerp_speed)

	TheCamera:SetGains(pan_gain, heading_gain, distance_gain)
end

local function StartBoatCamera(inst)
	local camera_settings =
	{
		state = {
			target_camera_offset = Vector3(0,1.5,0),
			camera_offset = Vector3(0,1.5,0),
			last_platform_x = 0, last_platform_z = 0,
			target_pan_gain = 4,
		},
		UpdateFn = BoatCam_UpdateFn,
		ActiveFn = BoatCam_ActiveFn,
		IsEnabled = BoatCam_IsEnabledFn,
	}

	TheFocalPoint.components.focalpoint:StartFocusSource(inst, nil, nil, math.huge, math.huge, -1, camera_settings)
end

local function OnSpawnNewBoatLeak(inst, data)
	if data ~= nil and data.pt ~= nil then
		local leak = SpawnPrefab("boat_leak")
		leak.Transform:SetPosition(data.pt:Get())
		leak.components.boatleak.isdynamic = true
		leak.components.boatleak:SetBoat(inst)
		leak.components.boatleak:SetState(data.leak_size)

		table.insert(inst.components.hullhealth.leak_indicators_dynamic, leak)

		if inst.components.walkableplatform ~= nil then
			for k,v in pairs(inst.components.walkableplatform:GetEntitiesOnPlatform()) do
				if v:IsValid() then
					v:PushEvent("on_standing_on_new_leak")
				end
			end
		end

		if data.playsoundfx then
			inst.SoundEmitter:PlaySoundWithParams("turnoftides/common/together/boat/damage", { intensity = 0.8 })
		end
	end
end

local function OnObjGotOnPlatform(inst, obj)    
    if obj == ThePlayer and inst.StartBoatCamera ~= nil then
		inst:StartBoatCamera()
	end
end

local function OnObjGotOffPlatform(inst, obj) 
    if obj == ThePlayer then
		TheFocalPoint.components.focalpoint:StopFocusSource(inst)
	end
end

local function RemoveConstrainedPhysicsObj(physics_obj)
	if physics_obj:IsValid() then
		physics_obj.Physics:ConstrainTo(nil)
		physics_obj:Remove()
	end
end

local function AddConstrainedPhysicsObj(boat, physics_obj)
	physics_obj:ListenForEvent("onremove", function() RemoveConstrainedPhysicsObj(physics_obj) end, boat)

	if boat:IsValid() then
		physics_obj.Transform:SetPosition(boat.Transform:GetWorldPosition())
		physics_obj.Physics:ConstrainTo(boat.entity)
	end
end

local function on_start_steering(inst)
	if ThePlayer and ThePlayer.components.playercontroller ~= nil and ThePlayer.components.playercontroller.isclientcontrollerattached then
		inst.components.reticule:CreateReticule()
	end
end

local function on_stop_steering(inst)
	if ThePlayer and ThePlayer.components.playercontroller ~= nil and ThePlayer.components.playercontroller.isclientcontrollerattached then
		inst.lastreticuleangle = nil
		inst.components.reticule:DestroyReticule()
	end
end

local function ReticuleTargetFn(inst)
	local range = 7
	local pos = Vector3(inst.Transform:GetWorldPosition())

	local dir = Vector3()
	dir.x = TheInput:GetAnalogControlValue(CONTROL_MOVE_RIGHT) - TheInput:GetAnalogControlValue(CONTROL_MOVE_LEFT)
	dir.y = 0
	dir.z = TheInput:GetAnalogControlValue(CONTROL_MOVE_UP) - TheInput:GetAnalogControlValue(CONTROL_MOVE_DOWN)
	local deadzone = .3 

	if math.abs(dir.x) >= deadzone or math.abs(dir.z) >= deadzone then
		dir = dir:GetNormalized()

		inst.lastreticuleangle = dir
	else
		if inst.lastreticuleangle then
			dir = inst.lastreticuleangle
		else 
			return nil
		end
	end

	local Camangle = TheCamera:GetHeading()/180
	local theta = -PI *(0.5 - Camangle)

	local newx = dir.x * math.cos(theta) - dir.z *math.sin(theta)
	local newz = dir.x * math.sin(theta) + dir.z *math.cos(theta)

	pos.x = pos.x - (newx * range)
	pos.z = pos.z - (newz * range)

	return pos
end

-----Attach collision mesh----
local function AttachCollision(boat, collision, offset_x, offset_z)
	boat:ListenForEvent("onremove", function() collision:Remove() end, boat)
	if collision.Update_Mesh ~= nil then
		collision:Update_Mesh(0, true)
	end
	collision.boat = boat
	collision.offset = {offset_x, offset_z}
	collision:AddTag("ignorewalkableplatforms")
	table.insert(boat.fixed_objects, collision)
	
	local ground_support = collision.ground_support
	if ground_support then
		boat:ListenForEvent("onremove", function() ground_support:Remove() end, boat)	
		ground_support:AddTag("ignorewalkableplatforms")
	end
	collision:DoTaskInTime(0, function()
		if boat:IsValid() then
			local x, y, z = boat.Transform:GetWorldPosition()
			local offset_x_r, offset_z_r = VecUtil_RotateDir(offset_x, offset_z , boat.rotation or 0)

			if ground_support then
				ground_support.Physics:ConstrainTo(nil)
			end
			collision.Transform:SetPosition(x + offset_x_r, y, z + offset_z_r)
			if ground_support then
				ground_support.Transform:SetPosition(x + offset_x_r, y, z + offset_z_r)
				ground_support.Physics:ConstrainTo(collision.entity)
			end			
			collision.Physics:SetActive(true)
		end
	end)
end

local function FinishRemovingEntity(entity)
	if entity:IsValid() then
		entity:Remove()
	end
end

local function AttachEntityToBoat(inst, obj, offset_x, offset_z, parent_to_boat, offset_y)
	obj:ListenForEvent("onremove", function() FinishRemovingEntity(obj) end, inst)

	if parent_to_boat then
		obj.entity:SetParent(inst.entity)
		obj.Transform:SetPosition(offset_x, 0, offset_z)	
	else
		inst:DoTaskInTime(0, function(boat)
			local boat_x, boat_y, boat_z = boat.Transform:GetWorldPosition()
			obj.Transform:SetPosition(boat_x + offset_x, boat_y + offset_y or 0, boat_z + offset_z)
		end)
	end
end

-----Rotation-----
local function SetRotation(inst, angle_d)
	-- angle in Degrees
	local angle = -angle_d * DEGREES

	local camangle = TheCamera:GetHeading()
	if not TheNet:IsDedicated() and angle ~= inst.rotation or camangle ~= inst.lip_lastcamangle then			
		for k, lip in pairs(inst.lip_fxs) do
			lip:Update(angle_d, camangle)
		end
		inst.lip_lastcamangle = camangle
	end

	if angle ~= inst.rotation then
		inst.vertices = rotate_vertices(inst.base_vertices, angle)
		if inst.itemcollision ~= nil then
			inst.itemcollision:Update_Mesh(angle)
		end 
		if inst.playercollision ~= nil then
			inst.playercollision:Update_Mesh(angle)
		end
		if inst.hullcollision ~= nil then
			inst.hullcollision:Update_Mesh(angle)
		end
		inst.rotation = angle
	end
end

----Map and walkableplatform Util----
local function is_point_on_boat(boat, pos_x, pos_y, pos_z, vertices)
	local boat_x, boat_y, boat_z = boat.Transform:GetWorldPosition()
	return is_point_in_poly(pos_x - boat_x, pos_z - boat_z, vertices or boat.vertices)
end

local function dist_to_border(boat, x, y, z)
	local bx, by, bz = boat.Transform:GetWorldPosition()
	local px, pz = closest_point_in_poly(x, z, bx, bz, boat.vertices)
	return VecUtil_Length(px - x, pz - z)
end

local function is_point_on_boat_with_bias(boat, x, y, z, bias)
	if bias == 0 then
		return is_point_on_boat(boat, x, y, z)
	elseif bias < 0 then
		if is_point_on_boat(boat, x, y, z) then
			return dist_to_border(boat, x, y, z) > bias
		else
			return false
		end
	elseif bias > 0 then
		if is_point_on_boat(boat, x, y, z) then
			return true
		else
			return dist_to_border(boat, x, y, z) < bias
		end
	end
end

local PLANT_SCALE = 1.1
local function Complete_data(data)
	-- calc automatically all collision meshes from vertices, for silly shape compute them yourself
	local vertices = data.vertices
	local bissectors = vertices_bissectors(vertices)
	local draw_vertices = data.draw_vertices or vertices
	local draw_bissectors = data.draw_vertices and vertices_bissectors(draw_vertices) or bissectors

	data.world_collider_size = 0.2
	data.vertices_world = subdivise(vertices, 1) -- collide with world and other boats
	data.vertices_player = shift_vertices(vertices, bissectors, 0.1, 1) -- collide with players
	data.vertices_hull = shift_vertices(vertices, bissectors, data.world_collider_size + 0.1) --collide with obstacles (seastack, etc) (0.1 with world_collider seem necessary)
	data.vertices_item = shift_vertices(vertices, bissectors, 0.2, 1) -- collide with items
	data.vertices_placer = subdivise(shift_vertices(vertices, bissectors, (2 * data.world_collider_size + 0.1) + 0.1), 0.5) --shape to have no object inside when deploying
	data.vertices_embarker = cut_reflex_angle(shift_vertices(draw_vertices, draw_bissectors, -0.55 * PLANT_SCALE), draw_bissectors, -0.55 * PLANT_SCALE) --where to land when embarking
	data.vertices_lip = subdivise(shift_vertices(vertices, bissectors, 0.1), 1.25) -- where to place boat_lip
	data.vertices_waveyjones = subdivise(data.vertices_embarker, 0.2) --where to spawn waveyjones hand its hands.
	if data.custom then
		data.vertices_plant = calc_edge_plant(vertices, bissectors, data.cuts, 0.55 * PLANT_SCALE, 1.6 * PLANT_SCALE)
		local inner = cut_reflex_angle(shift_vertices(draw_vertices, draw_bissectors, -0.275 * PLANT_SCALE), draw_bissectors, -0.275 * PLANT_SCALE) --half plant width
		data.plant_grid = make_grid(inner, 2 * PLANT_SCALE, 0.6 * PLANT_SCALE, true)
		data.burnable_locator = make_grid(inner, 3, 3)
		data.inner = inner
	end
end

-----Embarker Util-----
local function GetEmbarkPosition(inst, x, z)
	local bx, by, bz = inst.Transform:GetWorldPosition()
	local angle = -inst.Transform:GetRotation() * DEGREES
	local pos_x, pos_z = closest_point_in_poly(x, z, bx, bz, rotate_vertices(inst.vertices_embarker, angle))
	return pos_x, pos_z
end

local function Calcwaveyjonesposition(boat, x, z)
	local boat_x, boat_y, boat_z = boat.Transform:GetWorldPosition()
	local vertices = rotate_vertices(boat.vertices_waveyjones, -boat.Transform:GetRotation() * DEGREES)
	local min_dist, min_x, min_z, min_angle = nil
	local idx1 = #vertices
	for idx0 = 1, #vertices do
		local x0, z0 = unpack(vertices[idx0])
		local x1, z1 = unpack(vertices[idx1])
		local mid_x, mid_z = VecUtil_Lerp(x0, z0, x1, z1, 0.5)
		local dist = VecUtil_Length(boat_x + mid_x - x, boat_z + mid_z - z)
		local angle = VecUtil_GetAngleInDegrees(x1 - x0, z1 - z0)
		if min_dist == nil or dist < min_dist then
			min_dist, min_x, min_z, min_angle = dist, mid_x, mid_z, angle
		end
		idx1 = idx0
	end
	return boat_x + min_x, boat_z + min_z, min_angle
end

local DEPLOY_IGNORE_TAGS = { "NOBLOCK", "player", "FX", "INLIMBO", "DECOR"} -- from map
local function is_valid_placement(pt, inst, deploy_points, radius, only_land, extra_world_points, EXTRA_IGNORE_TAGS)
	--collide with world and boat
	for k, v in pairs(deploy_points) do
		if TheWorld.Map:IsPassableAtPointWithPlatformRadiusBias(pt.x + v[1], pt.y, pt.z + v[2], false, only_land, 0.25) then
			return false, k
		end
	end
	for k, v in pairs(extra_world_points or {}) do
		if TheWorld.Map:IsPassableAtPointWithPlatformRadiusBias(pt.x + v[1], pt.y, pt.z + v[2], false, only_land, 0.25) then
			return false, k
		end
	end
	if only_land then
		return true
	end
	if EXTRA_IGNORE_TAGS then
		DEPLOY_IGNORE_TAGS = {unpack(DEPLOY_IGNORE_TAGS)}
		for _, v in pairs(EXTRA_IGNORE_TAGS) do
			table.insert(DEPLOY_IGNORE_TAGS, v)
		end
	end
	--collide with items
	for i, v in ipairs(TheSim:FindEntities(pt.x, 0, pt.z, radius, nil, DEPLOY_IGNORE_TAGS)) do
		if v ~= inst and v.entity:IsVisible() and v.components.placer == nil and v.entity:GetParent() == nil then			
			local x, y, z = v.Transform:GetWorldPosition()
			if is_point_in_poly(x - pt.x, z - pt.z, deploy_points) then
				return false
			end
		end
	end
	return true
end

local function custom_candeploy_fn(inst, pt, mouseover, deployer)
	local angle = nil
	if inst.placer ~= nil then  --for client server
		angle = inst.placer.inst.Transform:GetRotation()
	elseif inst.deploy_rotation ~= nil then --for master server on action done
		angle = inst.deploy_rotation
	else  --for master server
		return true
	end
	local deploy_points = rotate_vertices(inst.vertices_placer, -angle * DEGREES)

	return (mouseover == nil or mouseover:HasTag("player")) and is_valid_placement(pt, inst, deploy_points, inst.deploy_radius)
end

local function post_init_placer(inst)
	local function SetBuilder(self, builder, recipe, invobject)		
		self.builder = builder
		invobject.placer = self
		self.recipe = recipe
		self.invobject = invobject
		self.inst:StartWallUpdatingComponent(self)
	end
	inst.components.placer.SetBuilder = SetBuilder
end

local function OnSave(inst, data)
	if inst.data then
		data.custom_data = inst.data
		data.send_vertices = inst.send_vertices
	end
end

local function OnPreLoad(inst, data)
	if data.custom_data then
		inst:post_fn(data.custom_data)		
	end
	if data.send_vertices then
		inst.send_vertices = data.send_vertices
		--cf prefabs/player_common.lua, OnPlayerJoined, line 494 ms_playerjoined
		-- necessary because boat_data netvars are created later on client, I guess
		TheWorld:ListenForEvent("ms_playerjoined", function() 
			inst:send_data(data.send_vertices)
		end)
	end
end

local function MakeBoat(data)
	local function post_fn(inst, data) --for data that cannot be changed after

		local placer_radius = CalcRadius(data.vertices_placer)
		if placer_radius > TUNING.MAX_WALKABLE_PLATFORM_RADIUS then
			TUNING.MAX_WALKABLE_PLATFORM_RADIUS = placer_radius
		end

		if data.custom then --to save/load
			inst.data = {}
			for k, v in pairs(data) do
				inst.data[k] = v
			end
		end

		inst:AddComponent("walkableplatform_common")
		inst.components.walkableplatform = inst.components.walkableplatform_common

		local max_health = data.health
		inst.components.healthsyncer.max_health = max_health

		inst.is_point_on_boat = is_point_on_boat
		inst.is_point_on_boat_with_bias = is_point_on_boat_with_bias

		inst.vertices = data.vertices --rotated
		inst.base_vertices = data.vertices --fixed
		inst.SetRotation = SetRotation

		inst.vertices_waveyjones = data.vertices_waveyjones
		inst.Calcwaveyjonesposition = Calcwaveyjonesposition

		inst.vertices_embarker = data.vertices_embarker
		inst.GetEmbarkPosition = GetEmbarkPosition

		inst.components.walkableplatform.platform_radius = CalcRadius(data.vertices) + 0.01
		inst.components.walkableplatform.radius = CalcRadius(data.vertices)

		if inst.components.walkableplatform.radius > 9 then
			inst.StartBoatCamera = nil
		end

		inst.fixed_objects = {}

		local itemcollision = SpawnPrefab("boat_item_collision_common")
		itemcollision.vertices = data.vertices_item
		AttachCollision(inst, itemcollision, 0, 0)
		inst.itemcollision = itemcollision

		local playercollision = SpawnPrefab("boat_player_collision_common")
		playercollision.vertices = data.vertices_player
		AttachCollision(inst, playercollision, 0, 0)
		playercollision.collisionboat = inst
		inst.playercollision = playercollision

		inst.lip_fxs = {}
		inst.plants = {}
		inst.edge_plants = {}

		inst:AddComponent("waterphysics")
		inst.components.waterphysics.restitution = 1.75

		inst:AddComponent("reticule")
		inst.components.reticule.targetfn = ReticuleTargetFn
		inst.components.reticule.ispassableatallpoints = true
		inst.on_start_steering = on_start_steering
		inst.on_stop_steering = on_stop_steering

		if not TheNet:IsDedicated() then
			if inst.components.walkableplatform.radius < 8.0 then
				-- dedicated server doesnt need to handle camera settings
				inst.StartBoatCamera = StartBoatCamera
				inst:ListenForEvent("obj_got_on_platform", OnObjGotOnPlatform)
				inst:ListenForEvent("obj_got_off_platform", OnObjGotOffPlatform)

				inst:ListenForEvent("endsteeringreticule", function(inst,data)  if ThePlayer and ThePlayer == data.player then inst:on_stop_steering() end end)
				inst:ListenForEvent("starsteeringreticule", function(inst,data) if ThePlayer and ThePlayer == data.player then inst:on_start_steering() end end)

				inst:AddComponent("boattrail")

				local THRESHOLD = 0.2
				inst:DoPeriodicTask(0.5,function()
					local pos = Vector3(inst.Transform:GetWorldPosition())
					if inst.oldpos then
						local diff = pos - inst.oldpos
						local lengthsq = diff:LengthSq()
						if lengthsq >= THRESHOLD and (not inst.oldspeed or inst.oldspeed < THRESHOLD) then
							local ents = inst.components.walkableplatform:GetEntitiesOnPlatform()
							for i,ent in ipairs(ents) do
								if ent == ThePlayer then
									ThePlayer:PushEvent("boatspedup")
								end
							end
						end
						inst.oldspeed = lengthsq
					end
					inst.oldpos = pos
				end)
			end

			if GetModConfigData("lip") then
				local idx1 = #data.vertices_lip
				for idx0 = 1, #data.vertices_lip do
					local x0, z0 = unpack(data.vertices_lip[idx0])
					local x1, z1 = unpack(data.vertices_lip[idx1])
					local mid_x, mid_z = VecUtil_Lerp(x0, z0, x1, z1, 0.54)
					local length = VecUtil_Length(x1 - x0, z1 - z0)
					local angle = VecUtil_GetAngleInDegrees(x1 - x0, z1 - z0)
					
					if length > 0.4 then
						local lip = SpawnPrefab("boatlip_fx")
						AttachEntityToBoat(inst, lip, mid_x, mid_z, true)
						lip.AnimState:SetScale(length / 1.4, 1, 1)
						lip.Transform:SetRotation(-angle)
						lip.AnimState:SetTime(math.random(0, 20) * FRAMES)
						lip.boat = inst

						table.insert(inst.lip_fxs, lip)
					end
					idx1 = idx0
				end
			end
		end

		--inst.entity:SetPristine()

		if not TheWorld.ismastersim then
			if data.postinitprefabfn ~= nil then
				data.postinitprefabfn(inst)
			end
			return
		end

		inst.loot = data.loot

		inst:AddComponent("hull")
		inst.components.hull:SetRadius(CalcRadius(data.vertices))

		inst.components.hull.OnDeployed = (function (self)
			if self.boat_lip ~= nil then
				self.boat_lip.AnimState:PlayAnimation("place_lip")
				self.boat_lip.AnimState:PushAnimation("lip", true)
			end
			if self.plank ~= nil then
				self.plank:Hide()
				self.plank:DoTaskInTime(1.25, function() self.plank:Show() self.plank.AnimState:PlayAnimation("plank_place") end)
			end
		end)

		inst:AddComponent("repairable")
		inst.components.repairable.repairmaterial = MATERIALS.WOOD
		inst.components.repairable.onrepaired = OnRepaired

		inst:AddComponent("hullhealth_common")
		inst.components.hullhealth = inst.components.hullhealth_common

		inst:AddComponent("boatphysics_common")
		inst.components.boatphysics = inst.components.boatphysics_common

		inst:AddComponent("health")
		inst.components.health:SetMaxHealth(max_health)
		inst.components.health.nofadeout = true

		inst.activefires = 0

		inst:SetStateGraph("SGboat_common")

		inst:ListenForEvent("spawnnewboatleak", OnSpawnNewBoatLeak)

		inst:AddComponent("savedrotation")
		inst.components.savedrotation.OnLoad = (function(self, data)
			self.inst.Transform:SetRotation(data.rotation or 0)
			self.inst:SetRotation(data.rotation or 0)
		end)

		for k, v in pairs(data.vertices_world) do
			local worldcollider = SpawnPrefab("boat_world_collision_common")
			worldcollider.Physics:SetCylinder(data.world_collider_size, 3)
			worldcollider.Physics:SetActive(false)
			AttachCollision(inst, worldcollider, v[1], v[2])
		end

		local hullcollision = SpawnPrefab("boat_hull_collision_common")
		hullcollision.vertices = data.vertices_hull
		AttachCollision(inst, hullcollision, 0, 0)
		inst.hullcollision = hullcollision


		if data.walkingplank ~= nil then
			local walking_plank = SpawnPrefab("walkingplank")
			inst.components.hull:AttachEntityToBoat(walking_plank, data.walkingplank[1], data.walkingplank[2], true)
			walking_plank.Transform:SetRotation(data.walkingplank[3])
			inst.components.hull:SetPlank(walking_plank)
		end

		for k, v in pairs(data.burnable_locator) do
			local burnable_locator = SpawnPrefab('burnable_locator_medium')
			burnable_locator.boat = inst
			inst.components.hull:AttachEntityToBoat(burnable_locator, v[1], v[2], true)
		end

		if data.plant_grid then
			local plant_anim = {}
			for k, v in pairs(data.plant_grid) do
				local plant = SpawnPrefab("boat_plant")
				plant.AnimState:SetScale(v[3] / 1.6, v[4] / 0.55 * 1.05)
				local anim
				if inst.data.plant_anim and inst.data.plant_anim[k] then
					anim = inst.data.plant_anim[k]
				else
					anim = weighted_random_choice({0.58, 0.27, 0.15}) + 2
				end
				plant.AnimState:PlayAnimation("idle_"..anim)
				table.insert(plant_anim, anim)
				plant.Transform:SetRotation(3.0 * (2 * math.random() - 1))
				AttachEntityToBoat(inst, plant, v[1], v[2], true)
				table.insert(inst.plants, plant)
			end
			inst.data.plant_anim = plant_anim
		end

		if data.vertices_plant then
			for k, s in pairs(data.vertices_plant) do
				local x0, z0, x1, z1 = unpack(s)
				local mid_x, mid_z = VecUtil_Lerp(x0, z0, x1, z1, 0.5)
				local length = VecUtil_Length(x1 - x0, z1 - z0)
				local angle = VecUtil_GetAngleInDegrees(x1 - x0, z1 - z0)
				local plant = SpawnPrefab("boat_plant")
				plant.AnimState:SetScale(length/1.6, 1 * PLANT_SCALE)
				plant.Transform:SetRotation(-angle + 180)
				plant.AnimState:SetLayer(LAYER_WORLD_BACKGROUND)
				plant.AnimState:PlayAnimation("idle_"..(k % 3))
				AttachEntityToBoat(inst, plant, mid_x, mid_z, true)
				table.insert(inst.edge_plants, plant)
			end
		end

		inst.maxspeedmultiplier = data.maxspeedmultiplier
		inst.rudderturnspeedmultiplier = data.rudderturnspeedmultiplier

		if data.postinitprefabfn ~= nil then
			data.postinitprefabfn(inst)
		end
	end

	local function fn() --for data that can be changed later
		local inst = CreateEntity()

		inst.entity:AddTransform()
		inst.entity:AddSoundEmitter()
		inst.entity:AddNetwork()

		inst.entity:AddAnimState()
		inst.AnimState:SetBank(data.bank)
		inst.AnimState:SetBuild(data.build)
		inst.AnimState:SetSortOrder(ANIM_SORT_ORDER.OCEAN_BOAT)
		inst.AnimState:SetFinalOffset(1)
		inst.AnimState:SetOrientation(ANIM_ORIENTATION.OnGround)
		inst.AnimState:SetLayer(LAYER_BACKGROUND)

		local phys = inst.entity:AddPhysics()
		phys:SetMass(TUNING.BOAT.MASS)
		phys:SetFriction(0)
		phys:SetDamping(5)
		phys:SetCollisionGroup(COLLISION.OBSTACLES)
		phys:ClearCollisionMask()
		phys:CollidesWith(COLLISION.GROUND)
		phys:SetCylinder(2, 2)
		phys:SetDontRemoveOnSleep(true)

		inst.entity:AddMiniMapEntity()
		inst.MiniMapEntity:SetIcon("boat.png")

		inst:AddTag("ignorewalkableplatforms")
		inst:AddTag("antlion_sinkhole_blocker")
		inst:AddTag("boat")
		inst:AddTag("wood")

		inst:AddComponent("healthsyncer") --contain net_var that mess up your, must be declared here

		inst.OnSave = OnSave
		inst.OnPreLoad = OnPreLoad

		if not data.custom then
			post_fn(inst, data)
			return inst
		end

		inst:AddComponent("boat_data_network")
		inst.vertices_length = net_uint(inst.GUID, "vertices.length", "vertices.length")
		inst.boat_data = net_entity(inst.GUID, "boat_data.entity", "boat_data.entity")

		inst.data = data

		inst.post_fn = post_fn
		inst.construct_data = data.construct_data

		inst.entity:SetPristine()

		if TheWorld.ismastersim then
			inst.entity:SetCanSleep(false)
		else
			inst:receive_data(function(draw_vertices)
				local _, _, _, _, _, area, vertices, cuts, radius, draw_vertices = is_constructible_poly(draw_vertices, 0.55 * PLANT_SCALE)
				local data = inst:construct_data(draw_vertices, vertices, cuts, area, radius)
				inst:post_fn(data)
			end)
		end
		return inst
	end

	table.insert(assets, Asset("ANIM", "anim/"..data.build..".zip"))
	return Prefab(data.prefabname, fn, assets, prefabs)
end

local function MakeKit(data)
	local placer_radius = CalcRadius(data.vertices_placer)

	local function ondeploy(inst, pt, deployer, rot)
		local boat = SpawnPrefab(data.prefabname)
		if boat ~= nil then
			boat.Physics:SetCollides(false)
			boat.Physics:Teleport(pt.x, 0, pt.z)
			boat.Transform:SetRotation(rot)
			boat.components.boatphysics:SetRotation(-rot * DEGREES)
			boat.Physics:SetCollides(true)

			boat.sg:GoToState("place")

			boat.components.hull:OnDeployed()

			inst:Remove()
		end
	end

	local function item_fn()
		local inst = CreateEntity()

		inst.entity:AddTransform()
		inst.entity:AddAnimState()
		inst.entity:AddNetwork()

		inst:AddTag("boatbuilder")
		inst:AddTag("usedeployspacingasoffset")

		MakeInventoryPhysics(inst)

		inst.AnimState:SetBank(data.itembank)
		inst.AnimState:SetBuild(data.itembuild)
		inst.AnimState:PlayAnimation("IDLE")

		MakeInventoryFloatable(inst, "med", 0.25, 0.83)

		inst._custom_candeploy_fn = custom_candeploy_fn
		inst.deploy_radius = placer_radius
		inst.vertices_placer = data.vertices_placer

		inst:AddComponent("deployable")
		inst.components.deployable.ondeploy = ondeploy
		inst.components.deployable.DeploySpacingRadius = function(self) return placer_radius - 0.5 end
		inst.components.deployable:SetDeployMode(DEPLOYMODE.CUSTOM)

		inst.entity:SetPristine()

		if not TheWorld.ismastersim then
			return inst
		end

		inst:AddComponent("inspectable")
		inst:AddComponent("inventoryitem")
		if data.itemasset == nil then
			inst.components.inventoryitem.atlasname = "images/inventoryimages/"..data.item_prefabname..".xml"
		end

		inst:AddComponent("fuel")
		inst.components.fuel.fuelvalue = TUNING.LARGE_FUEL

		MakeLargeBurnable(inst)
		MakeLargePropagator(inst)
		MakeHauntableLaunch(inst)

		return inst
	end

	table.insert(item_assets, Asset("ANIM", "anim/"..data.itembuild..".zip"))

	if data.itemasset ~= nil then
		table.insert(item_assets, data.itemasset)
	else
		table.insert(item_assets, Asset("ATLAS", "images/inventoryimages/"..data.item_prefabname..".xml"))
		table.insert(item_assets, Asset("IMAGE", "images/inventoryimages/"..data.item_prefabname..".tex"))
	end

	return Prefab(data.item_prefabname, item_fn, item_assets, item_prefabs),
		MakePlacer(data.item_prefabname.."_placer", data.bank, data.build, "idle_full", true,    false, false,     nil,   90 ,               nil,    post_init_placer, placer_radius)
				   -- name,							bank,      build,      anim,        onground,snap,  metersnap, scale, fixedcameraoffset, facing, postinit_fn,      offset,        onfailedplacement
end

return {MakeBoat, MakeKit, Complete_data, AddRecipe, GetModConfigData, is_valid_placement, PLANT_SCALE}
