local env = getfenv(1)
for k, v in pairs(require('boat_mathutil')) do env[k] = v end
for k, v in pairs(require('prefabs/shipyard_common')) do env[k] = v end
setfenv(1, env)
local _, _, _, AddRecipe, GetModConfigData = unpack(require("prefabs/boat_common"))

local assets =
{
	Asset("ANIM", "anim/shipyard_flag.zip"),
	Asset("ATLAS", "images/inventoryimages/shipyard_flag_item.xml"),
	Asset("IMAGE", "images/inventoryimages/shipyard_flag_item.tex")
}

----Shipyard----
local FIND_DIST = 8
local PLACER_DIST = 14

local function Spawn_ropes(flag1, flag2, old_ropes, placer)
	local x1, y1, z1 = flag1.Transform:GetWorldPosition()
	local x2, y2, z2 = flag2.Transform:GetWorldPosition()
	local length = VecUtil_Length(x2 - x1, z2 - z1)
	local angle = -VecUtil_GetAngleInDegrees(x2 - x1, z2 - z1)
	local n_step = math.ceil(length / 1.7)

	local ropes = {}
	for i = 0, n_step - 1 do
		local rope = table.remove(old_ropes, 1) or SpawnPrefab("shipyard_rope")
		local x, z = VecUtil_Lerp(x1, z1, x2, z2, (i + 0.5)/ n_step)
		rope.Transform:SetPosition(x, 0, z)
		rope.AnimState:SetScale(length / n_step / 1.73, 1)
		rope.Transform:SetRotation(angle)
		if placer then
			rope.AnimState:SetAddColour(0, 0.5, 0, 0)
		else
			rope.flag:set(flag1)
		end
		table.insert(ropes, rope)
	end
	return ropes
end

local function Link(flag1, flag2)
	-- flag can be nil
	if flag1 then
		for k, v in pairs(flag1.ropes) do
			v:Remove()
		end
		flag1.next_flag:set(flag2)
		if flag2 and flag2 ~= flag1 then
			 flag1.ropes = Spawn_ropes(flag1, flag2, {})
		end
		TheWorld.components.shipyardmanager:UnregisterShipyard(flag1)
	end
	if flag2 then
		flag2.prev_flag:set(flag1)
		TheWorld.components.shipyardmanager:UnregisterShipyard(flag2)
	end
end

local function OnUpdate_placer(inst)
	if inst:IsValid() then
		inst.prev_flag, inst.next_flag = Closest_flags({inst.Transform:GetWorldPosition()}, FIND_DIST)
		local old_ropes = inst.ropes
		local new_ropes = {}
		if inst.prev_flag and inst.components.placer.can_build then
			for k, rope in pairs(Spawn_ropes(inst, inst.prev_flag, old_ropes, true)) do
				table.insert(new_ropes, rope)
			end
		end
		if inst.next_flag and inst.components.placer.can_build then
			for k, rope in pairs(Spawn_ropes(inst, inst.next_flag, old_ropes, true)) do
				table.insert(new_ropes, rope)
			end
		end
		for k, rope in pairs(old_ropes) do
			rope:Remove()
		end
		inst.ropes = new_ropes
	end
end

local function GetRopes(inst)
	return TheWorld.components.shipyardmanager:GetRegisteredRopes(inst) or {}
end

local function custom_candeploy_fn(inst, pt, mouseover, deployer)
	return TheWorld.Map:CanDeployAtPointInWater(pt, inst, mouseover,
        {land = 0.6, boat = 0.3})
end

local function ondeploy(inst, pt, deployer, rot)
	local prev_flag, next_flag = Closest_flags({pt.x, 0, pt.z}, FIND_DIST)

	local flag = SpawnPrefab("shipyard_flag")
	flag.Transform:SetPosition(pt.x, 0, pt.z)

	if prev_flag and next_flag then
		Link(prev_flag, flag)
		Link(flag, next_flag)
	else
		Link(flag, flag)
	end

	if inst.components.stackable ~= nil then
		inst.components.stackable:Get():Remove()
	else
		inst:Remove()
	end
end

local PLACER_RING_SCALE = 0.5
local function CreateDeployHelperRing()
    local inst = CreateEntity()

    inst.entity:SetCanSleep(false)
    inst.persists = false

    inst.entity:AddTransform()
    inst.entity:AddAnimState()

    inst:AddTag("CLASSIFIED")
    inst:AddTag("NOCLICK")
    inst:AddTag("placer")

    inst.AnimState:SetBank("winona_battery_placement")
    inst.AnimState:SetBuild("winona_battery_placement")
    inst.AnimState:PlayAnimation("idle_small")
    inst.AnimState:SetAddColour(0, 0.5, 0, 0)
    inst.AnimState:SetLightOverride(1)
    inst.AnimState:SetOrientation(ANIM_ORIENTATION.OnGround)
    inst.AnimState:SetLayer(LAYER_BACKGROUND)
    inst.AnimState:SetSortOrder(3)
    inst.AnimState:SetScale(PLACER_RING_SCALE, PLACER_RING_SCALE)

    return inst
end

local function OnEnableHelper(inst, enabled, recipename, placerinst)
	local function DeployHelperRing_OnUpdate(helperinst)
		helperinst.parent.highlited = nil
		if not helperinst.placerinst:IsValid() then
			helperinst.components.updatelooper:RemoveOnUpdateFn(DeployHelperRing_OnUpdate)
		else
			if (inst == placerinst.prev_flag or inst == placerinst.next_flag) and placerinst.components.placer.can_build then
				return helperinst:Show()
			else
				helperinst:Hide()
			end
		end
	end

	if enabled then
		if inst.helper == nil and not inst:HasTag("burnt") then
			inst.helper = CreateDeployHelperRing()
			inst.helper.parent = inst
			inst.helper.entity:SetParent(inst.entity)
			if placerinst ~= nil then
				inst.helper:AddComponent("updatelooper")
				inst.helper.components.updatelooper:AddOnUpdateFn(DeployHelperRing_OnUpdate)
				inst.helper.placerinst = placerinst
				DeployHelperRing_OnUpdate(inst.helper)
			end
		end
	elseif inst.helper ~= nil then
		inst.helper:Remove()
		inst.helper = nil
	end
end

local function onprotect(inst)
	--prevent to be destroyed or to be linked again
    if inst.components.burnable ~= nil then
		if inst.components.burnable:IsBurning() then
        	inst.components.burnable:Extinguish()
		end
		inst.components.burnable.canlight = false
    end
	for _, rope in pairs(inst.ropes) do
		rope:RemoveTag("shipyard_rope")
	end
	inst:AddTag("fireimmune")
	inst:RemoveTag("shipyard_flag")
	inst.AnimState:OverrideSymbol("flag", "shipyard_flag", "flag_green")
end

local function onunprotect(inst)
	if inst and inst:IsValid() then
		if inst.components.burnable then
			inst:RemoveTag("fireimmune")
			inst.components.burnable.canlight = true
		end
		for _, rope in pairs(inst.ropes) do
			rope:AddTag("shipyard_rope")
		end
		inst:AddTag("shipyard_flag")
		inst.AnimState:ClearOverrideSymbol("flag")
	end
end

local function candismantle(inst)
	return inst:HasTag("shipyard_flag")
end

local function OnSave(inst, data)
	if inst.next_flag:value() then
		data.next_flag_pos = {inst.next_flag:value().Transform:GetWorldPosition()}
	end
end

local function OnLoad(inst, data)
	inst:DoTaskInTime(0, function()
		if TheWorld.ismastersim and data and data.next_flag_pos then
			local x, y, z = unpack(data.next_flag_pos)
			local ents = TheSim:FindEntities(x, y, z, 0.01, SHIPYARD_FLAGS)
			if #ents > 0 then
				Link(inst, ents[1])
			else
				Link(inst, inst)
			end
		end
	end)
end

local function Onremove(inst)	
	TheWorld.components.shipyardmanager:UnregisterShipyard(inst)
	local prev_flag, next_flag = inst.prev_flag:value(), inst.next_flag:value()
	for _, rope in pairs(inst.ropes) do
		rope:Remove()
	end
	if TheWorld.ismastersim then
		Link(prev_flag, next_flag)
	end
end

local function OnDismantle(inst, doer)
	if doer.components.inventory ~= nil and inst:IsValid() and inst:HasTag("shipyard_flag") then
		doer.components.inventory:GiveItem(SpawnPrefab("shipyard_flag_item"))
		inst:Remove()
	end
end

local function getstatus(inst)
	return not inst:HasTag("shipyard_flag") and "PROTECTED"
end

local function fn()
    local inst = CreateEntity()

    inst.entity:AddTransform()
    inst.entity:AddAnimState()
    inst.entity:AddNetwork()

	inst:AddTag("ignorewalkableplatforms_motion")
	inst:AddTag("ignorewalkableplatformdrowning")
	inst:AddTag("shipyard_flag")
	inst:AddTag("NOBLOCK")

	inst.AnimState:SetBank("shipyard_flag")
    inst.AnimState:SetBuild("shipyard_flag")
    inst.AnimState:PlayAnimation("idle", true)
	inst.AnimState:SetTime(math.random() * 1.5)

	MakeInventoryFloatable(inst, "small", -0.1, 0.7)
	inst.components.floater.bob_percent = 0

	inst.prev_flag = net_entity(inst.GUID, "shipyard_flag.prev_flag", "receive_prev_flag")
	inst.next_flag = net_entity(inst.GUID, "shipyard_flag.next_flag")

	inst.GetRopes = GetRopes
	inst.candismantle = candismantle

	inst.physicsradiusoverride = PLACER_DIST + 0.7 --increase dismantle distance

	if not TheNet:IsDedicated() then
		inst:AddComponent("deployhelper")
		inst.components.deployhelper:AddKeyFilter("shipyard_flag_placer")
		inst.components.deployhelper.onenablehelper = OnEnableHelper
		inst:ListenForEvent("receive_prev_flag", function() TheWorld.components.shipyardmanager:UnregisterShipyard(inst) end)
    end
	inst.OnSave = OnSave
	inst.OnLoad = OnLoad

	inst.ropes = {}

	inst:ListenForEvent("onremove", Onremove)
	inst.protect = onprotect
	inst.unprotect = onunprotect

 	inst.entity:SetPristine()

	if not TheWorld.ismastersim then
        return inst
    end

	inst.entity:SetCanSleep(false) --necessary to keep shipyard linked on client

    inst:AddComponent("inspectable")
	inst.components.inspectable.getstatus = getstatus

    inst:AddComponent("portablestructure")
    inst.components.portablestructure:SetOnDismantleFn(OnDismantle)

	MakeSmallBurnable(inst)

	inst:ListenForEvent("got_on_platform", function(inst) inst:Hide() end)
	inst:ListenForEvent("got_off_platform", function(inst) inst.components.floater:OnLandedServer() inst:Show() end)

	inst:DoTaskInTime(0, function(inst)
        inst.components.floater:OnLandedServer()
		if not inst.next_flag:value() then --when spawned with console
			Link(inst, inst)
		end
    end)

	return inst
end

local function item_fn()
    local inst = CreateEntity()

    inst.entity:AddTransform()
    inst.entity:AddAnimState()
    inst.entity:AddNetwork()

    MakeInventoryPhysics(inst)
	MakeInventoryFloatable(inst, "med", 0, 0.8)

	inst:AddTag("portableitem")
 	inst:AddTag("usedeployspacingasoffset")

    inst.AnimState:SetBank("shipyard_flag")
    inst.AnimState:SetBuild("shipyard_flag")
    inst.AnimState:PlayAnimation("item")

	inst:AddComponent("deployable")
	inst.components.deployable.ondeploy = ondeploy
	inst.components.deployable.DeploySpacingRadius = function(self) return PLACER_DIST - 1 end
	inst.components.deployable:SetDeployMode(DEPLOYMODE.CUSTOM)
	inst._custom_candeploy_fn = custom_candeploy_fn

    inst.entity:SetPristine()

    if not TheWorld.ismastersim then
        return inst
    end

    inst:AddComponent("inspectable")

	inst:AddComponent("inventoryitem")
	inst.components.inventoryitem.atlasname = "images/inventoryimages/shipyard_flag_item.xml"
	inst.components.inventoryitem:ChangeImageName("shipyard_flag_item")
	inst.components.inventoryitem:EnableMoisture(false)

	MakeHauntableLaunchAndIgnite(inst)

    inst:AddComponent("stackable")
    inst.components.stackable.maxsize = TUNING.STACK_SIZE_MEDITEM

    return inst
end

local function placer_postinit_fn(inst)
	inst.deployhelper_key = "shipyard_flag_placer"
	inst:AddComponent("updatelooper")
	inst.ropes = {}
	inst.components.updatelooper:AddOnUpdateFn(OnUpdate_placer)
	inst:ListenForEvent("onremove", function(inst)
		for k, rope in pairs(inst.ropes) do
			rope:Remove()
		end
	end)
end

STRINGS.NAMES["SHIPYARD_FLAG"] = "Shipyard flag"
STRINGS.NAMES["SHIPYARD_FLAG_ITEM"] = "Shipyard Flag"
STRINGS.RECIPE_DESC["SHIPYARD_FLAG_ITEM"] = "Draw the boat of your dreams."
STRINGS.CHARACTERS.GENERIC.DESCRIBE["SHIPYARD_FLAG_ITEM"] = "It marks the edge of my future boat."
STRINGS.CHARACTERS.GENERIC.DESCRIBE["SHIPYARD_FLAG"] = 
	{
		GENERIC = "I need to check the shape before building",
		PROTECTED = "Just a few boards and it's done."
	}

AddRecipe("shipyard_flag_item", {Ingredient("petals", 1), Ingredient("twigs", 3), Ingredient("rope", 1)}, RECIPETABS.SEAFARING, TECH.SEAFARING_TWO, nil, nil, nil, 8, nil, "images/inventoryimages/shipyard_flag_item.xml")

return Prefab("shipyard_flag", fn, assets),
	   Prefab("shipyard_flag_item", item_fn, assets),
	   MakePlacer("shipyard_flag_item_placer", "shipyard_flag", "shipyard_flag", "idle", nil, nil, nil, nil, nil, nil, placer_postinit_fn)
